//=============================================================================
// Grid Inventory
// MRP_GridInventory.js
// Version: Beta 1.10
// By Magnus0808 || MagnusRubin || Magnus Rubin Peterson
//=============================================================================

/* Change log
* Beta 1.10     : Added Compare Items Window
*                 Added DisassembleItems Preview (for YEP_X_ItemDisassemble)
*                 Added plugin command to clear player inventory.
*           	  Equip Windows now properly shows tool tips.   
*                 You will now drop items on the floor if they cannot fit in your inventory
* Beta 1.9.5	: Changed how custom image icons are drawn
*                 Changed default Inventory Scene to show two actors instead of four
* Beta 1.9		: Added WindowHighlight (used to show which window uses the Keyboard)
*				  Added InformationSprite to the ItemDrop scene
*				  Reworked how you equip equipment using the keyboard
* Beta 1.8		: Added a lot of customization for the Equip Windows in the player
*				  inventory.
*				  Added Names/Headers to the Inventories
*				  A lot of bug fixes
* Beta 1.7      : Added even more customization options for the tool tip window.
*                 including Notetags! 
* Beta 1.6		: Added Confirmation Window when dropping/destroying items
*				  Added a lot of customization options for the tool tip window
*				  Added customization options for the item command windows
*				  The held item now uses a HeldItemManager
*    			  Bug Fixes
* Beta 1.5      : Made the plugin compatible with more YEP_ItemCore extensions
*                      - YEP_X_AttachAugments
*                      - YEP_X_ItemUpgrades
*                 Added missing stuff to the tool tip window
* Beta 1.4		: Made the plugin compatible with various YEP_ItemCore extensions
*                      - YEP_X_ItemDisassemble (Limited)
*					   - YEP_X_ItemDurability (missing tool tip info)
*                      - YEP_X_ItemRename
*				  Changed various behind the scenes stuff for handling mouse and keyboard
* Beta 1.3      : Added customization options for item & cursor background colors
*                 Added Rarity system to customize item background/text colors
* Beta 1.2      : Added plugin command for openning the custom inventory
*                 Fixed the custom inventory scene when using the keyboard
* Beta 1.1.1    : Fixed usage of items from the inventory
*                 Fixed hiding the tooltip window
* Beta 1.1		: Changed some logic behind adding items to the inventory.
*				  Made the plugin compatible with YEP_ItemCore
* Beta 1.0.1 	: Fixed Equip Windows bug when not in a party of 4.
* Beta 1.0 		: Initial Beta Release
*/

/*:
 * @plugindesc Changes the way the player interacts with their inventory
 * @author Magnus0808
 *
 * @help
 * You can use the following meta-tags to modify how items behave in the inventory.
 * <gridSize:x,y> - replace x and y with how much space the item should take (default 1,1)
 * <maxItems:x> - replace x with the max amount of items the item can stack (default 1)
 * <iconImage:imgName> - replace imgName with the image name of the image. This will make the item
 * use the selected image file as the icon instead in the grid.
 * <Rarity:x> - replace x with the ID of the rarity that is setup in the plugin parameters.
 * 
 * You can also customize the player inventory/item grids to some degree by changing the plugin 
 * parameters.
 * 
 * To change the inventory size of the player inventory change the amount of player columns and rows.
 * However it should be noted that if the inventory is too large in size (e.g. player rows * cell size
 * is too large) The inventory scene will not work correctly.
 * 
 * You can also change the cell size in the parameters. At the moment cells are always squares because
 * of the way icons are drawn.
 * 
 * ///////// Highlighted Windows /////////
 * The plugin requires that you have another Window skin which is used to highlight windows.
 * This window skin should also be placed in /img/system and should be called 'WindowHighlight.png'.
 * 
 * /////////////// Rarity //////////////// 
 * You can define the different rarities in the plugin parameters. The rarity is used to
 * change background color and/or text colors of an item.
 * You can set the rarity of an item using the meta-tag:
 *      <Rarity:x> - replace x with the ID of the rarity
 * If an item do not have a rarity defined then it will used the first rarity defined in the 
 * plugin parameters as the default one.
 * 
 * //////////// Icon Images //////////////
 * As you might have noticed you can use custom images as icons for items. However, instead of loading
 * individual images this plugin requires the images to be in a spritesheet. However, since the icons
 * might vary in size this plugin also requires a data file containing information about the icons. 
 * This data file alongwith the spritesheet can easily be made using a program such as TexturePacker. 
 * You can find a quick tutorial how to do this on the itch.io page for this plugin.
 * 
 * The data file should be put in the data folder of your project, while the spritesheet should be placed
 * in ./img/system.
 * 
 * /////////// Equip Windows /////////////
 * You can now customize the equip windows quite a lot in the plugin parameters. Under 'Window Types' you
 * can specify how a Equipment Window looks like. You have to give the type a name which will work as an ID.
 * Then under 'Equip Slots' you can specify the size and location of each slot.
 * 
 * You can then use the following notetag to specify that a class should use a equipment window type:
 * 
 * Class Notetag:
 * 
 * 	  <equipType:X>
 * 
 * - Replace X with the name of the equiptype
 *  E.g <equipType:Default>
 * 
 * ///////////// Tool Tip ///////////////
 * The following note tags can be used to used to control the tooltip window for specific objects:
 * 
 * Item/Weapon/Armor Notetags:
 * 
 *    <ToolTipVisibility:TAG, true/false>
 * or for setting multiple tags at once:
 *    <ToolTipVisibility>
 *       TAG: true/false
 *       TAG: true/false
 *    </ToolTipVisibility>
 * 
 * - Replace TAG with the parameter code for the tag. You can find the TAG in the plugin parameters.
 *   e.g. the TAG for the title is showTitle
 * 
 *     <ToolTipFormat:TAG,formatText>
 * or for setting multiple tags at once:
 *     <ToolTipFormat>
 *        Tag:formatText
 *        Tag:formatText
 *     </ToolTipFormat>
 * - Replace TAG with the parameter code for the tag. You can find the TAG in the plugin parameters.
 *   e.g. the TAG for the title is showTitle
 *   Replace formatText with the text you want shown. 
 * 
 * @param Equip Windows
 * 
 * @param Only Show Battle Members
 * @parent Equip Windows
 * @type Boolean
 * @desc Should the equip windows only be shown for battle members
 * instead of all party members?
 * @default false
 * 
 * @param Number of Equipment Windows
 * @parent Equip Windows
 * @type Number
 * @desc The number of windows that should be room for.
 * This determines the width of the windows
 * @default 2
 * 
 * @param equipWindowTypes
 * @text Window Types
 * @parent Equip Windows
 * @type struct<EquipWindowType>[]
 * @desc The equip window type.
 * @default ["{\"name\":\"Default\",\"equipSlots\":\"[\\\"{\\\\\\\"x\\\\\\\":\\\\\\\"this.contentsWidth() - 6.5 * cell\\\\\\\",\\\\\\\"y\\\\\\\":\\\\\\\"2.5 * cell\\\\\\\",\\\\\\\"width\\\\\\\":\\\\\\\"2 * cell\\\\\\\",\\\\\\\"height\\\\\\\":\\\\\\\"4 * cell\\\\\\\"}\\\",\\\"{\\\\\\\"x\\\\\\\":\\\\\\\"this.contentsWidth() - 1.5 * cell\\\\\\\",\\\\\\\"y\\\\\\\":\\\\\\\"2.5 * cell\\\\\\\",\\\\\\\"width\\\\\\\":\\\\\\\"2 * cell\\\\\\\",\\\\\\\"height\\\\\\\":\\\\\\\"4 * cell\\\\\\\"}\\\",\\\"{\\\\\\\"x\\\\\\\":\\\\\\\"this.contentsWidth() - 4 * cell\\\\\\\",\\\\\\\"y\\\\\\\":\\\\\\\"0.5 * cell\\\\\\\",\\\\\\\"width\\\\\\\":\\\\\\\"2 * cell\\\\\\\",\\\\\\\"height\\\\\\\":\\\\\\\"2 * cell\\\\\\\"}\\\",\\\"{\\\\\\\"x\\\\\\\":\\\\\\\"this.contentsWidth() - 4 * cell\\\\\\\",\\\\\\\"y\\\\\\\":\\\\\\\"3 * cell\\\\\\\",\\\\\\\"width\\\\\\\":\\\\\\\"2 * cell\\\\\\\",\\\\\\\"height\\\\\\\":\\\\\\\"3 * cell\\\\\\\"}\\\",\\\"{\\\\\\\"x\\\\\\\":\\\\\\\"this.contentsWidth() - cell\\\\\\\",\\\\\\\"y\\\\\\\":\\\\\\\"cell\\\\\\\",\\\\\\\"width\\\\\\\":\\\\\\\"cell\\\\\\\",\\\\\\\"height\\\\\\\":\\\\\\\"cell\\\\\\\"}\\\"]\",\"customCode\":\"\\\"var infoWidth = w / 4;\\\\n// Draw Divider\\\\nthis.contents.paintOpacity = 48;\\\\nthis.contents.fillRect(infoWidth + 12, 0, 1, this.contentsHeight(), this.normalColor());\\\\nthis.contents.paintOpacity = 255;\\\\n\\\\nvar y = 0.5 * cell;\\\\n// Draw Name\\\\nthis.drawText(a.name(), 0, y, infoWidth, 'center');\\\\ny += this.lineHeight();\\\\n\\\\n// Draw Face\\\\nthis.drawActorFace(a, 0, y, infoWidth, infoWidth);\\\\ny += infoWidth;\\\\n\\\\n// Draw Level\\\\nthis.changeTextColor(this.systemColor());\\\\nthis.drawText(TextManager.levelA, 0, y, 48);\\\\nthis.resetTextColor();\\\\nthis.drawText(a.level, 24, y, infoWidth - 24, 'right');\\\\ny += this.lineHeight();\\\\n\\\\n// Draw HP\\\\nthis.drawActorHp(a, 0, y, infoWidth);\\\\ny += this.lineHeight();\\\\n\\\\n// Draw MP\\\\nthis.drawActorMp(a, 0, y, infoWidth);\\\\ny += this.lineHeight();\\\\n\\\\n// Draw TP\\\\nthis.drawActorTp(a, 0, y, infoWidth);\\\"\"}"]
 * 
 * @param Player Columns
 * @type Number
 * @desc The amount of columns the player inventory  has
 * @default 10
 * 
 * @param Player Rows
 * @type Number
 * @desc The amount of rows the player inventory has
 * @default 6
 * 
 * @param Cell Size
 * @type Number
 * @desc The size of each cell in a grid.
 * @default 36
 * 
 * @param Icon Padding
 * @type Number
 * @desc The padding used when drawing icons in a cell
 * @default 4
 * 
 * @param Colors
 * 
 * @param Item Background Rarity
 * @parent Colors
 * @type Boolean
 * @desc Should the background color of items be based on their rarity instead?
 * @default false
 * 
 * @param Usable Background Color
 * @parent Colors
 * @type String
 * @desc The color of the background when selecting an item (Default: #58D3F7)
 * @default #58D3F7
 * 
 * @param Unusable Background Color
 * @parent Colors
 * @type String
 * @desc The color of the cursor when selecting an item (Default: #58D3F7)
 * @default #58D3F7
 * 
 * @param Select Color
 * @parent Colors
 * @type String
 * @desc The color of the cursor when selecting an item (Default: #58D3F7)
 * @default #58D3F7
 * 
 * @param Usable Color
 * @parent Colors
 * @type String
 * @desc The color of the cursor when you can place an item (Default: #2EFE2E)
 * @default #2EFE2E
 * 
 * @param Unusable Color
 * @parent Colors
 * @type String
 * @desc The color of the cursor when you cannot place an item (Default: #FF0000)
 * @default #FF0000
 * 
 * @param Swap Color
 * @parent Colors
 * @type String
 * @desc The color of the cursor when you can swap items (Default: #FFBF00)
 * @default #FFBF00
 * 
 * @param Inventories
 * @type struct<ItemDrops>[]
 * @desc These are the inventories you can place around the world! 
 * @default []
 * 
 * @param Rarities
 * @type struct<Rarity>[]
 * @desc The rarities used in the game. Currently mainly used for color options.
 * @default ["{\"Id\":\"Normal\",\"Color\":\"#FFFFFF\"}","{\"Id\":\"Magical\",\"Color\":\"#3090C7\"}","{\"Id\":\"Rare\",\"Color\":\"#FFD801\"}","{\"Id\":\"Unique\",\"Color\":\"#C58917\"}"]
 * 
 * @param Confirm Drop Window Config
 * @type struct<WindowCommandOptions>
 * @desc The config for the confirm drop of item window.
 * @default {"textTitle":"Destroy Item","textBody":"Are you sure you want to destroy the item?","textConfirm":"Yes","textCancel":"No","width":"240"}
 * 
 * @param Select Item Command Window Config
 * @type struct<SelectItemWindow>
 * @desc The config for the command window when selecting an item
 * @default {"textMove":"Move","textDrop":"Drop","textUseItem":"Use","textEquipEquipment":"Equip"}
 *
 * @param Use Item Command Window Config
 * @type struct<WindowItemHeldCommand>
 * @desc The config for the command window when using an item
 * @default {"textSwap":"Swap","textPlace":"Place"}
 *
 * @param Tool Tip Config
 * @type struct<ToolTipConfig>
 * @desc The config for the tool tip
 * @default {"showTitle":"true","formatTitle":"%1","showType":"true","formatType":"%1","showMaxItems":"true","formatMaxItems":"Stack size: %1/%2","showDamage":"true","formatDamage":"[\"Deals %1 %2 Damage\",\"Removes %1 Mana\",\"Heals %1 Health\",\"Restores %1 Mana\",\"Drain %1 Health\",\"Drain %1 Mana\"]","showDescription":"true","Yanfly Expansions":"","showDurability":"true","showUpgrades":"true","showAugments":"true","showItemEffects":"true","effectHelper":"{\"textLose\":\"Lose \",\"textRecover\":\"Recover \",\"textBoth\":\" and \"}","showEffect11":"true","formatEffect11":"%3%1%5%4%2 Health","showEffect12":"true","formatEffect12":"%3%1%5%4%2 Mana","showEffect13":"true","formatEffect13":"%4%2 TP","showEffect21":"true","formatEffect21":"%2 chance to gain %1","showEffect22":"true","formatEffect22":"%2 chance to remove %1","showEffect31":"true","formatEffect31":"Add %1 buff for %2 turns","showEffect32":"true","formatEffect32":"Add %1 debuff for %2 turns","showEffect33":"true","formatEffect33":"Remove %1 buff for %2 turns","showEffect34":"true","formatEffect34":"Remove %1 debuff for %2 turns","showEffect41":"true","formatEffect41":"Escape from battle","showEffect42":"true","formatEffect42":"Gain %2 %1","showEffect43":"true","formatEffect43":"Learn the %1 skill","showParameters":"true","formatParameters":"%2 %1","showMaxHP":"true","showMaxMP":"true","showAttack":"true","showDefense":"true","showMAttack":"true","showMDefense":"true","showAgility":"true","showLuck":"true","showTraits":"true","showTrait11":"true","formatTrait11":"%2 %1 Resistance","showTrait12":"true","formatTrait12":"%2 %1 Debuff Resistance","showTrait13":"true","formatTrait13":"%2 %1 Resistance","showTrait14":"true","formatTrait14":"Gain %1 Immunity","showTrait21":"true","formatTrait21":"%2 %1","showTrait22":"true","formatTrait22":"%2 %1","showTrait23":"true","formatTrait23":"%2 %1","showTrait31":"true","formatTrait31":"Add %1 Element to Normal Attacks","showTrait32":"true","formatTrait32":"%2 Chance to Apply %1","showTrait33":"true","formatTrait33":"%1 Attack Speed","showTrait34":"true","formatTrait34":"Attack %1 More Times","showTrait41":"true","formatTrait41":"Gain %1 Skill Type","showTrait42":"true","formatTrait42":"Seal %1 SKill Type","showTrait43":"true","formatTrait43":"Add %1 Skill","showTrait44":"true","formatTrait44":"Seal %1 SKill","showTrait51":"true","formatTrait51":"Equip the %1 Weapon","showTrait52":"true","formatTrait52":"Equip the %1 Armor","showTrait53":"true","formatTrait53":"Lock the %1 Equip Type","showTrait54":"true","formatTrait54":"Seal the %1 Equip Type","showTrait55":"true","formatTrait55":"Has the %1 Slot Type","showTrait61":"true","formatTrait61":"Gives a %1 Chance to Gain an Extra Action","showTrait62":"true","formatTrait62":"[\"The User Will Lose Control in Battle\", \"Decrases the amount of damage taken by a fixed rate\", \"Take damage in place of an ally when their HP is low.\", \"Preserves TP\"]"}
 *
 */

/*~struct~EquipWindowType:
 * @param name
 * @text Name
 * @type String
 * @desc The name/id of the window type
 * 
 * @param equipSlots
 * @text Equip Slots
 * @type struct<EquipSlot>[]
 * @desc The EquipSlots making up the window
 * @default
 * 
 * @param customCode
 * @text Custom Code
 * @type Note
 * @desc Custom code to be run when creating the window.
 * Can be used to add custom text etc.
 * @default ""
 * 
 */

 /*~struct~EquipSlot:
 * @param x
 * @type String
 * @desc The x-coordinat of the slot
 * 
 * @param y
 * @type String
 * @desc the y-coordinat of the slot
 * 
 * @param width
 * @type String
 * @desc The width of the slot
 * @default this.
 * 
 * @param height
 * @type String
 * @desc The height of the slot
 * 
 */

/*~struct~ToolTipConfig:
 * @param showTitle
 * @text Show Item Name
 * @type Boolean
 * @desc Should the item name be shown?
 * @default true
 * 
 * @param formatTitle
 * @text Item Name Text
 * @type String
 * @desc The text for the Item name
 * %1 - Item Name
 * @default %1
 *
 * @param showType
 * @text Show Item Type
 * @type Boolean
 * @desc Should the item type be shown?
 * @default true
 * 
 * @param formatType
 * @text Type Text
 * @type String
 * @desc The text for the Type name
 * %1 - Type Text
 * @default %1
 * 
 * @param showMaxItems
 * @text Show Max Items
 * @type Boolean
 * @desc Should the max items be shown?
 * @default true
 * 
 * @param formatMaxItems
 * @text Max Items Text
 * @type String
 * @desc The text for Max Items
 * %1 - Current Amount. %2 - Max Amount
 * @default Stack size: %1/%2
 * 
 * @param showDamage
 * @text Show Damage
 * @type Boolean
 * @desc Should the damage be shown?
 * @default true
 * 
 * @param formatDamage
 * @text Damage Text
 * @type String[]
 * @desc The text for Damage. Each entry corrosponds to a Type.
 * %1 - Formular. %2 - Element
 * @default ["Deals %1 %2 Damage","Removes %1 Mana","Heals %1 Health","Restores %1 Mana","Drain %1 Health","Drain %1 Mana"]
 * 
 * @param showDescription
 * @text Show Description
 * @type Boolean
 * @desc Should the description be shown?
 * @default true
 * 
 * @param Yanfly Expansions
 * 
 * @param showDurability
 * @parent Yanfly Expansions
 * @text Show the durability of equipment
 * @type Boolean
 * @desc should the durability of equipment be shown? (Requires YEP_X_ItemDurability)
 * @default true
 * 
 * @param showUpgrades
 * @parent Yanfly Expansions
 * @text Show the upgrades of equipment
 * @type Boolean
 * @desc should the upgrades of equipment be shown? (Requires YEP_X_ItemUpgrades)
 * @default true
 * 
 * @param showAugments
 * @parent Yanfly Expansions
 * @text Show the augments of equipment
 * @type Boolean
 * @desc should the augments of equipment be shown? (Requires YEP_X_AttachAugments)
 * @default true 
 * 
 * @param showItemEffects
 * @text Show Effects for Items
 * @type Boolean
 * @desc Should any of the effects be shown?
 * @default true
 * 
 * @param effectHelper
 * @parent showItemEffects
 * @text Commands for Effects
 * @type struct<EffectValueText>
 * @desc These are the words for %3, %4, and %5
 * @default {"textLose1":"Lose","textRecover1":"Recover","textLose2":"Lose","textRecover2":"Recover","textBoth":"and "}
 * 
 * @param showEffect11
 * @parent showItemEffects
 * @text Show the 'Recover HP' effect
 * @type Boolean
 * @desc should the 'Recover HP' effect be shown?
 * @default true
 * 
 * @param formatEffect11
 * @parent showItemEffects
 * @text 'Recover HP' text
 * @type String
 * @desc The text for 'Recover HP'
 * %1 - % value. %2 - flat value. %3, %4, %5
 * @default %3%1%5%4%2 Health
 * 
 * @param showEffect12
 * @parent showItemEffects
 * @text Show the 'Recover MP' effect
 * @type Boolean
 * @desc should the 'Recover MP' effect be shown?
 * @default true
 * 
 * @param formatEffect12
 * @parent showItemEffects
 * @text 'Recover MP' text
 * @type String
 * @desc The text for 'Recover MP'
 * %1 - % value. %2 - flat value. %3, %4, %5
 * @default %3%1%5%4%2 Mana
 * 
 * @param showEffect13
 * @parent showItemEffects
 * @text Show the 'Recover TP' effect
 * @type Boolean
 * @desc should the 'Recover TP' effect be shown?
 * @default true
 * 
 * @param formatEffect13
 * @parent showItemEffects
 * @text 'Recover TP' text
 * @type String
 * @desc The text for 'Recover TP'
 * %2 - flat value. %4 - Recover/Lose
 * @default %4%2 TP
 * 
 * @param showEffect21
 * @parent showItemEffects
 * @text Show the 'Add State' effect
 * @type Boolean
 * @desc should the 'Add State' effect be shown?
 * @default true
 * 
 * @param formatEffect21
 * @parent showItemEffects
 * @text 'Add State' text
 * @type String
 * @desc The text for 'Add State'
 * %1 - State. %2 - Percent
 * @default %2 chance to gain %1
 * 
 * @param showEffect22
 * @parent showItemEffects
 * @text Show the 'Remove State' effect
 * @type Boolean
 * @desc should the 'Remove State' effect be shown?
 * @default true
 * 
 * @param formatEffect22
 * @parent showItemEffects
 * @text 'Remove State' text
 * @type String
 * @desc The text for 'Remove State'
 * %1 - State. %2 - Percent
 * @default %2 chance to remove %1
 * 
 * @param showEffect31
 * @parent showItemEffects
 * @text Show the 'Add Buff' effect
 * @type Boolean
 * @desc should the 'Add Buff' effect be shown?
 * @default true
 * 
 * @param formatEffect31
 * @parent showItemEffects
 * @text 'Add Buff' text
 * @type String
 * @desc The text for 'Add Buff'
 * %1 - Stat. %2 - Turns
 * @default Add %1 buff for %2 turns
 * 
 * @param showEffect32
 * @parent showItemEffects
 * @text Show the 'Add Debuff' effect
 * @type Boolean
 * @desc should the 'Add Debuff' effect be shown?
 * @default true
 * 
 * @param formatEffect32
 * @parent showItemEffects
 * @text 'Add Debuff' text
 * @type String
 * @desc The text for 'Add Debuff'
 * %1 - Stat. %2 - Turns
 * @default Add %1 debuff for %2 turns
 * 
 * @param showEffect33
 * @parent showItemEffects
 * @text Show the 'Remove Buff' effect
 * @type Boolean
 * @desc should the 'Remove Buff' effect be shown?
 * @default true
 * 
 * @param formatEffect33
 * @parent showItemEffects
 * @text 'Remove Buff' text
 * @type String
 * @desc The text for 'Remove Buff'
 * %1 - Stat. %2 - Turns
 * @default Remove %1 buff for %2 turns
 * 
 * @param showEffect34
 * @parent showItemEffects
 * @text Show the 'Remove Debuff' effect
 * @type Boolean
 * @desc should the 'Remove Debuff' effect be shown?
 * @default true
 * 
 * @param formatEffect34
 * @parent showItemEffects
 * @text 'Remove Debuff' text
 * @type String
 * @desc The text for 'Remove Debuff'
 * %1 - Stat. %2 - Turns
 * @default Remove %1 debuff for %2 turns
 * 
 * @param showEffect41
 * @parent showItemEffects
 * @text Show the 'Special Effect' effect
 * @type Boolean
 * @desc should the 'Special Effect' effect be shown?
 * @default true
 * 
 * @param formatEffect41
 * @parent showItemEffects
 * @text 'Special Effect' text
 * @type String
 * @desc The text for 'Special Effect'
 * @default Escape from battle
 * 
 * @param showEffect42
 * @parent showItemEffects
 * @text Show the 'Grow' effect
 * @type Boolean
 * @desc should the 'Grow' effect be shown?
 * @default true
 * 
 * @param formatEffect42
 * @parent showItemEffects
 * @text 'Grow' text
 * @type String
 * @desc The text for 'Grow'
 * %1 - Stat. %2 - Amount
 * @default Gain %2 %1
 * 
 * @param showEffect43
 * @parent showItemEffects
 * @text Show the 'Learn Skill' effect
 * @type Boolean
 * @desc should the 'Learn Skill' effect be shown?
 * @default true
 * 
 * @param formatEffect43
 * @parent showItemEffects
 * @text 'Learn Skill' text
 * @type String
 * @desc The text for 'Learn Skill'
 * %1 - Skill
 * @default Learn the %1 skill
 * 
 * @param showParameters
 * @text Show Parameters for Items
 * @type Boolean
 * @desc Should any of the parameters be shown?
 * @default true
 * 
 * @param formatParameters
 * @parent showParameters
 * @text Parameters Text
 * @type String
 * @desc The text for Parameters
 * %1 - Param. %2 - value
 * @default %2 %1
 * 
 * @param showMaxHP
 * @parent showParameters
 * @text Show the 'Max HP' parameter
 * @type Boolean
 * @desc should the 'Max HP' parameter be shown?
 * @default true
 * 
 * @param showMaxMP
 * @parent showParameters
 * @text Show the 'Max MP' parameter
 * @type Boolean
 * @desc should the 'Max MP' parameter be shown?
 * @default true
 * 
 * @param showAttack
 * @parent showParameters
 * @text Show the 'Attack' parameter
 * @type Boolean
 * @desc should the 'Attack' parameter be shown?
 * @default true
 * 
 * @param showDefense
 * @parent showParameters
 * @text Show the 'Defense' parameter
 * @type Boolean
 * @desc should the 'Defense' parameter be shown?
 * @default true
 * 
 * @param showMAttack
 * @parent showParameters
 * @text Show the 'M.Attack' parameter
 * @type Boolean
 * @desc should the 'M.Attack' parameter be shown?
 * @default true
 * 
 * @param showMDefense
 * @parent showParameters
 * @text Show the 'M.Defense' parameter
 * @type Boolean
 * @desc should the 'M.Defense' parameter be shown?
 * @default true
 * 
 * @param showAgility
 * @parent showParameters
 * @text Show the 'Agility' parameter
 * @type Boolean
 * @desc should the 'Agility' parameter be shown?
 * @default true
 * 
 * @param showLuck
 * @parent showParameters
 * @text Show the 'Luck' parameter
 * @type Boolean
 * @desc should the 'Luck' parameter be shown?
 * @default true
 * 
 * @param showTraits
 * @text Show Traits for Equipment
 * @type Boolean
 * @desc Should any of the traits be shown?
 * @default true
 * 
 * @param showTrait11
 * @parent showTraits
 * @text Show the 'Element Rate' trait
 * @type Boolean
 * @desc should the 'Element Rate' trait be shown?
 * @default true
 * 
 * @param formatTrait11
 * @parent showTraits
 * @text 'Element Rate' Text
 * @type String
 * @desc The text for 'Element Rate'
 * %1 - Element. %2 - Percentage
 * @default %2 %1 Resistance 
 * 
 * @param showTrait12
 * @parent showTraits
 * @text Show the 'Debuff Rate' trait
 * @type Boolean
 * @desc should the 'Debuff Rate' trait be shown?
 * @default true
 * 
 * @param formatTrait12
 * @parent showTraits
 * @text 'Debuff Rate' Text
 * @type String
 * @desc The text for 'Debuff Rate'
 * %1 - Stat. %2 - Percentage
 * @default %2 %1 Debuff Resistance 
 * 
 * @param showTrait13
 * @parent showTraits
 * @text Show the 'State Rate' trait
 * @type Boolean
 * @desc should the 'State Rate' trait be shown?
 * @default true
 * 
 * @param formatTrait13
 * @parent showTraits
 * @text 'State Rate' Text
 * @type String
 * @desc The text for 'State Rate'
 * %1 - State. %2 - Percentage
 * @default %2 %1 Resistance 
 * 
 * @param showTrait14
 * @parent showTraits
 * @text Show the 'State Resist' trait
 * @type Boolean
 * @desc should the 'State Resist' trait be shown?
 * @default true
 * 
 * @param formatTrait14
 * @parent showTraits
 * @text 'State Resist' Text
 * @type String
 * @desc The text for 'State Resist'
 * %1 - State
 * @default Gain %1 Immunity
 * 
 * @param showTrait21
 * @parent showTraits
 * @text Show the 'Parameter' trait
 * @type Boolean
 * @desc should the 'Parameter' trait be shown?
 * @default true
 * 
 * @param formatTrait21
 * @parent showTraits
 * @text 'Parameter' Text
 * @type String
 * @desc The text for 'Parameter'
 * %1 - Param. %2 - Percentage
 * @default %2 %1
 * 
 * @param showTrait22
 * @parent showTraits
 * @text Show the 'Ex-Parameter' trait
 * @type Boolean
 * @desc should the 'Ex-Parameter' trait be shown?
 * @default true
 * 
 * @param formatTrait22
 * @parent showTraits
 * @text 'Ex-Parameter' Text
 * @type String
 * @desc The text for 'Ex-Parameter'
 * %1 - Ex-Param. %2 - Percentage
 * @default %2 %1
 * 
 * @param showTrait23
 * @parent showTraits
 * @text Show the 'Sp-Parameter' trait
 * @type Boolean
 * @desc should the 'Sp-Parameter' trait be shown?
 * @default true
 * 
 * @param formatTrait23
 * @parent showTraits
 * @text 'Sx-Parameter' Text
 * @type String
 * @desc The text for 'Sp-Parameter'
 * %1 - Sp-Param. %2 - Percentage
 * @default %2 %1
 * 
 * @param showTrait31
 * @parent showTraits
 * @text Show the 'Attack Element' trait
 * @type Boolean
 * @desc should the 'Attack Element' trait be shown?
 * @default true
 * 
 * @param formatTrait31
 * @parent showTraits
 * @text 'Attack Element' Text
 * @type String
 * @desc The text for 'Attack Element'
 * %1 - Element
 * @default Add %1 Element to Normal Attacks
 * 
 * @param showTrait32
 * @parent showTraits
 * @text Show the 'Attack State' trait
 * @type Boolean
 * @desc should the 'Attack State' trait be shown?
 * @default true
 * 
 * @param formatTrait32
 * @parent showTraits
 * @text 'Attack State' Text
 * @type String
 * @desc The text for 'Attack State'
 * %1 - State. %2 - Percentage
 * @default %2 Chance to Apply %1
 * 
 * @param showTrait33
 * @parent showTraits
 * @text Show the 'Attack Speed' trait
 * @type Boolean
 * @desc should the 'Attack Speed' trait be shown?
 * @default true
 * 
 * @param formatTrait33
 * @parent showTraits
 * @text 'Attack Speed' Text
 * @type String
 * @desc The text for 'Attack Speed'
 * %1 - Value
 * @default %1 Attack Speed
 * 
 * @param showTrait34
 * @parent showTraits
 * @text Show the 'Attack Times +' trait
 * @type Boolean
 * @desc should the 'Attack Times +' trait be shown?
 * @default true
 * 
 * @param formatTrait34
 * @parent showTraits
 * @text 'Attack Times +' Text
 * @type String
 * @desc The text for 'Attack Times +'
 * %1 - Value
 * @default Attack %1 More Times
 * 
 * @param showTrait41
 * @parent showTraits
 * @text Show the 'Add Skill Type' trait
 * @type Boolean
 * @desc should the 'Add Skill Type' trait be shown?
 * @default true
 * 
 * @param formatTrait41
 * @parent showTraits
 * @text 'Add Skill Type' Text
 * @type String
 * @desc The text for 'Add Skill Type'
 * %1 - Skill Type
 * @default Gain %1 Skill Type
 * 
 * @param showTrait42
 * @parent showTraits
 * @text Show the 'Seal Skill Type' trait
 * @type Boolean
 * @desc should the 'Seal Skill Type' trait be shown?
 * @default true
 * 
 * @param formatTrait42
 * @parent showTraits
 * @text 'Seal Skill Type' Text
 * @type String
 * @desc The text for 'Seal Skill Type'
 * %1 - SKill Type
 * @default Seal %1 SKill Type
 * 
 * @param showTrait43
 * @parent showTraits
 * @text Show the 'Add Skill' trait
 * @type Boolean
 * @desc should the 'Add Skill' trait be shown?
 * @default true
 * 
 * @param formatTrait43
 * @parent showTraits
 * @text 'Add Skill' Text
 * @type String
 * @desc The text for 'Add Skill'
 * %1 - Skill
 * @default Add %1 Skill
 * 
 * @param showTrait44
 * @parent showTraits
 * @text Show the 'Seal Skill' trait
 * @type Boolean
 * @desc should the 'Seal Skill' trait be shown?
 * @default true
 * 
 * @param formatTrait44
 * @parent showTraits
 * @text 'Seal Skill' Text
 * @type String
 * @desc The text for 'Seal Skill'
 * %1 - Skill
 * @default Seal %1 SKill
 * 
 * @param showTrait51
 * @parent showTraits
 * @text Show the 'Equip Weapon' trait
 * @type Boolean
 * @desc should the 'Equip Weapon' trait be shown?
 * @default true
 * 
 * @param formatTrait51
 * @parent showTraits
 * @text 'Equip Weapon' Text
 * @type String
 * @desc The text for 'Equip Weapon'
 * %1 - Weapon
 * @default Equip the %1 Weapon
 * 
 * @param showTrait52
 * @parent showTraits
 * @text Show the 'Equip Armor' trait
 * @type Boolean
 * @desc should the 'Equip Armor' trait be shown?
 * @default true
 * 
 * @param formatTrait52
 * @parent showTraits
 * @text 'Equip Armor' Text
 * @type String
 * @desc The text for 'Equip Armor'
 * %1 - Armor
 * @default Equip the %1 Armor
 * 
 * @param showTrait53
 * @parent showTraits
 * @text Show the 'Lock Equip' trait
 * @type Boolean
 * @desc should the 'Lock Equip' trait be shown?
 * @default true
 * 
 * @param formatTrait53
 * @parent showTraits
 * @text 'Lock Equip' Text
 * @type String
 * @desc The text for 'Lock Equip'
 * %1 - Equip Type
 * @default Lock the %1 Equip Type
 * 
 * @param showTrait54
 * @parent showTraits
 * @text Show the 'Seal Equip' trait
 * @type Boolean
 * @desc should the 'Seal Equip' trait be shown?
 * @default true
 * 
 * @param formatTrait54
 * @parent showTraits
 * @text 'Seal Equip' Text
 * @type String
 * @desc The text for 'Seal Equip'
 * %1 - Equip Type
 * @default Seal the %1 Equip Type
 * 
 * @param showTrait55
 * @parent showTraits
 * @text Show the 'Slot Type' trait
 * @type Boolean
 * @desc should the 'Slot Type' trait be shown?
 * @default true
 * 
 * @param formatTrait55
 * @parent showTraits
 * @text 'Slot Type' Text
 * @type String
 * @desc The text for 'Slot Type'
 * %1 - Slot Type
 * @default Has the %1 Slot Type
 * 
 * @param showTrait61
 * @parent showTraits
 * @text Show the 'Action Times +' trait
 * @type Boolean
 * @desc should the 'Action Times +' trait be shown?
 * @default true
 * 
 * @param formatTrait61
 * @parent showTraits
 * @text 'Action Times +' Text
 * @type String
 * @desc The text for 'Action Times +'
 * %1 - Percentage
 * @default Gives a %1 Chance to Gain an Extra Action
 * 
 * @param showTrait62
 * @parent showTraits
 * @text Show the 'Special Flag' trait
 * @type Boolean
 * @desc should the 'Special Flag' trait be shown?
 * @default true
 * 
 * @param formatTrait62
 * @parent showTraits
 * @text 'Action Times +' Text
 * @type String[]
 * @desc The text for 'Action Times +'
 * @default ["The User Will Lose Control in Battle", "Decrases the amount of damage taken by a fixed rate", "Take damage in place of an ally when their HP is low.", "Preserves TP"]
 * 
 */

  /*~struct~WindowItemHeldCommand:
 * @param textSwap
 * @text Swap Item Text
 * @type String
 * @desc The swap command text
 * @default Swap
 *
 * @param textPlace
 * @text Place Item Text
 * @type String
 * @desc The place command text
 * @default Place
 */

  /*~struct~SelectItemWindow:
 * @param textMove
 * @text Move Text
 * @type String
 * @desc The move command text
 * @default Move
 *
 * @param textDrop
 * @text Drop Text
 * @type String
 * @desc The drop command text
 * @default Drop
 * 
 * @param textUseItem
 * @text Use Item Text
 * @type String
 * @desc The use item command text
 * @default Use
 * 
 * @param textEquipEquipment
 * @text Equip Equipment Text
 * @type String
 * @desc The equip equipment command text
 * @default Equip
 */


  /*~struct~WindowCommandOptions:
 * @param textTitle
 * @text Title Text
 * @type String
 * @desc The title text
 * @default Title Text
 *
 * @param textBody
 * @text Body Text
 * @type String
 * @desc The body text
 * @default Body Text
 * 
 * @param textConfirm
 * @text Confirm Text
 * @type String
 * @desc The text used for the 'positive' option
 * @default Confirm
 * 
 * @param textCancel
 * @text Cancel Text
 * @type String
 * @desc The text used for the 'negative' option
 * @default Cancel
 * 
 * @param width
 * @text Width
 * @type Number
 * @desc The width of the window
 * @default 240
 * 
 */

  /*~struct~EffectValueText:
 * @param textLose
 * @type String
 * @desc If negative value.
 * 
 * @param textRecover
 * @type String
 * @desc If positive value.
 *
 * @param textBoth
 * @type String
 * @desc Text if both %1 and %2 exist.
 * %1 - Percentage Value. %2 - Flat Value
 */

 /*~struct~Rarity:
 * @param Id
 * @type String
 * @desc The id used to identify the rarity
 *
 * @param Color
 * @type String
 * @desc The color associated with the rarity
 * @default #FFFFFF
 */

 /*~struct~ItemDrops:
 * @param Name
 * @type String
 * @desc The name of the inventory.
 * @default Drops
 *
 * @param Columns
 * @type Number
 * @default 10
 *
 * @param Rows
 * @type Number
 * @default 6
 * 
 * @param Items
 * @type Item[]
 * @default []
 *
 * @param Weapons
 * @type Weapon[]
 * @default []
 * 
 * @param Armors
 * @type Armor[]
 * @default []
 */

var Imported = Imported || {};
Imported.MRP_GridInventory = true;
 
var MRP = MRP || {};
MRP.GridInventory = MRP.GridInventory ||{};
 
MRP.GridInventory.Parameters = PluginManager.parameters('MRP_GridInventory');
MRP.GridInventory.playerCols = Number(MRP.GridInventory.Parameters['Player Columns']);
MRP.GridInventory.playerRows = Number(MRP.GridInventory.Parameters['Player Rows']);
MRP.GridInventory.cellSize = Number(MRP.GridInventory.Parameters['Cell Size']);
MRP.GridInventory.iconPadding = Number(MRP.GridInventory.Parameters['Icon Padding']);
MRP.GridInventory.useItemBackGroundRarity = String(MRP.GridInventory.Parameters['Item Background Rarity']) == 'true';
MRP.GridInventory.usableBackgroundColor = String(MRP.GridInventory.Parameters['Usable Background Color']);
MRP.GridInventory.unusableBackgroundColor = String(MRP.GridInventory.Parameters['Unusable Background Color']);
MRP.GridInventory.selectColor = String(MRP.GridInventory.Parameters['Select Color']);
MRP.GridInventory.usableColor = String(MRP.GridInventory.Parameters['Usable Color']);
MRP.GridInventory.unusableColor = String(MRP.GridInventory.Parameters['Unusable Color']);
MRP.GridInventory.swapColor = String(MRP.GridInventory.Parameters['Swap Color']);
MRP.GridInventory.dataInventories = JSON.parse(MRP.GridInventory.Parameters['Inventories']).map(e => JSON.parse(e)).map(function(e){
	e.Columns = Number(e.Columns);
	e.Rows = Number(e.Rows);
	e.Items = JSON.parse(e.Items).map(e => Number(e));;
	e.Weapons = JSON.parse(e.Weapons).map(e => Number(e));
	e.Armors = JSON.parse(e.Armors).map(e => Number(e));;
	return e;
});

MRP.GridInventory.onlyShowBattleMembers = String(MRP.GridInventory.Parameters["Only Show Battle Members"]) == "true"
MRP.GridInventory.numberOfEquipmentWindows = Number(MRP.GridInventory.Parameters["Number of Equipment Windows"]);
MRP.GridInventory.equipWindowTypes  = JSON.parse(MRP.GridInventory.Parameters["equipWindowTypes"]).map(e=>JSON.parse(e)).map(function(e){
	e.equipSlots = JSON.parse(e.equipSlots).map(e => JSON.parse(e));
	e.customCode = e.customCode != undefined ? JSON.parse(e.customCode) : "";
	return e;
});

MRP.GridInventory.confirmDropWindowConfig = JSON.parse(MRP.GridInventory.Parameters['Confirm Drop Window Config']);
MRP.GridInventory.confirmDropWindowConfig.width = Number(MRP.GridInventory.confirmDropWindowConfig.width);

MRP.GridInventory.selectItemWindowConfig = JSON.parse(MRP.GridInventory.Parameters['Select Item Command Window Config']);
MRP.GridInventory.useItemWindowConfig = JSON.parse(MRP.GridInventory.Parameters['Use Item Command Window Config']);
MRP.GridInventory.toolTipConfig = JSON.parse(MRP.GridInventory.Parameters['Tool Tip Config']);
for (k in MRP.GridInventory.toolTipConfig){
	var v = MRP.GridInventory.toolTipConfig[k]
	MRP.GridInventory.toolTipConfig[k] = (v == 'true' ? true : (v == 'false' ? false : v));
	if(v.startsWith("[") || v.startsWith("{")){
		MRP.GridInventory.toolTipConfig[k] = JSON.parse(v);
	}
}


MRP.GridInventory.rarities = {};
JSON.parse(MRP.GridInventory.Parameters['Rarities']).map(e => JSON.parse(e)).map(function(e){
	if(MRP.GridInventory.defaultRarity == undefined) MRP.GridInventory.defaultRarity = e;
	MRP.GridInventory.rarities[e.Id] = e;
	return e;
});


MRP.GridInventory.dropInventories = {};

MRP.GridInventory.makeEventData = function(eventId, inventoryId, x, y) {
	return {
		"id":eventId,
		"name":"EV0" + eventId,
		"note":"","pages":[{"conditions":{"actorId":1,"actorValid":false,"itemId":1,"itemValid":false,"selfSwitchCh":"A","selfSwitchValid":false,"switch1Id":1,"switch1Valid":false,"switch2Id":1,"switch2Valid":false,"variableId":1,"variableValid":false,"variableValue":0},"directionFix":true,
		"image":{"tileId":0,"characterName":"!Flame","direction":2,"pattern":0,"characterIndex":7},
		"list":[{
				"code":355, // Script
				"indent":0,
				"parameters":["$gameTemp._itemDropGrid = MRP.GridInventory.dropInventories['" + inventoryId + "'];"]
			},{
				"code":655, // Script next line
				"indent":0,
				"parameters":["SceneManager.push(Scene_ItemDrop);"]
			},{
				"code":111, // If
				"indent":0,
				"parameters":[12,"MRP.GridInventory.dropInventories['" + inventoryId + "'].isEmpty()"]
			}, {
				"code":214, // Erase
				"indent":1,
				"parameters":[]
			},{
				"code":0, // Empty
				"indent":1,
				"parameters":[]
			},{
				"code":412, // End
				"indent":0,
				"parameters":[]
			},{
				"code":0, // Empty
				"indent":0,
				"parameters":[]
			}],
			"moveFrequency":3,"moveRoute":{"list":[{"code":0,"parameters":[]}],"repeat":true,"skippable":false,"wait":false},"moveSpeed":3,"moveType":0,"priorityType":0,"stepAnime":true,"through":false,"trigger":0,"walkAnime":true}],
			"x":x,
			"y":y
	}
}

//-----------------------------------------------------------------------------
// Game_Interpreter
//
// Changes to Game_Interpreter
MRP.GridInventory.Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
Game_Interpreter.prototype.pluginCommand = function(command, args) {
	MRP.GridInventory.Game_Interpreter_pluginCommand.call(this, command, args)
	
	if (command === 'OpenInventory'){
		if(args[0].toLowerCase() == 'player'){
			SceneManager.push(Scene_Inventory);
		} else {
			var id = Number(args[0]);
			$gameTemp._itemDropGrid = MRP.GridInventory.inventories[id - 1];
			SceneManager.push(Scene_ItemDrop);
		}
	} else if (command === 'ClearPlayerInventory') {
		$gameParty.clearInventory();
	}
};

//-----------------------------------------------------------------------------
// Scene_Boot
//
// Changes to Scene_Boot

MRP.GridInventory.Scene_Boot_loadSystemImages =Scene_Boot.loadSystemImages
Scene_Boot.loadSystemImages = function() {
	MRP.GridInventory.Scene_Boot_loadSystemImages.call(this);
	ImageManager.reserveSystem('InventoryIcons');
};

//-----------------------------------------------------------------------------
// BattleManager
//
// Changes to BattleManager

MRP.GridInventory.BattleManager_gainDropItems = BattleManager.gainDropItems;
BattleManager.gainDropItems = function() {		
	this.makeDropChestEvent();
};

BattleManager.makeDropChestEvent = function(){
	if(this._rewards.items.length > 0) $gameMap.makeDropEvent(this._rewards.items);
	//$gameMap._events.push(new Game_DropEvent($gameMap.events().length + 1, $gameMap._mapId, this._rewards.items));
}

//-----------------------------------------------------------------------------
// TouchInput
//
// changes to TouchInput
MRP.GridInventory.TouchInput__onMouseMove = TouchInput._onMouseMove;
TouchInput._onMouseMove = function(event) {
	MRP.GridInventory.TouchInput__onMouseMove.call(this, event);
	this._events._mouseMoved = true;
	this._mouseX = Graphics.pageToCanvasX(event.pageX);
	this._mouseY = Graphics.pageToCanvasY(event.pageY);
};

MRP.GridInventory.TouchInput_update = TouchInput.update;
TouchInput.update = function() {
	MRP.GridInventory.TouchInput_update.call(this);
	this._mouseMoved = this._events._mouseMoved;
	this._anyMousePressed = this._events._anyMousePressed;
	this._events._mouseMoved = false;
	this._events._anyMousePressed = false;
};

MRP.GridInventory.TouchInput__onMouseDown = TouchInput._onMouseDown;
TouchInput._onMouseDown = function(event) {
	MRP.GridInventory.TouchInput__onMouseDown.call(this, event);
	this._events._anyMousePressed = true;
};

//-----------------------------------------------------------------------------
// Game_Party
//
// changes to Game_Party
Game_Party.prototype.gridItems = function() {
	return this._grid;
};

MRP.GridInventory.Game_Party_initAllItems = Game_Party.prototype.initAllItems;
Game_Party.prototype.initAllItems = function() {
	MRP.GridInventory.Game_Party_initAllItems.call(this);
	this._grid = new Game_PlayerGrid("Inventory", MRP.GridInventory.playerCols, MRP.GridInventory.playerRows);
};

Game_Party.prototype.clearInventory = function() {
	this._items = {};
	this._weapons = {};
	this._armors = {};

	this._grid.initGrid();
}

Game_Party.prototype.canEquip = function(item) {
	return this.members().some(function(actor) {
		return actor.canEquip(item);
	});
};
	
Game_Party.prototype.gainItem = function(item, amount, includeEquip) {
	if(Imported.YEP_ItemCore){
		if (DataManager.isIndependent(item)) {
			this.gainIndependentItem(item, amount, includeEquip);
			return;
		}
	}


	var container = this.itemContainer(item);
	if (container) {
		var lastNumber = this.numItems(item);
		var newNumber = lastNumber + amount;
		container[item.id] = newNumber;
		this._grid.pushItem(item, amount);
		// Clean up
		if (container[item.id] === 0) {
			delete container[item.id];
		}
		if (includeEquip && newNumber < 0) {
			this.discardMembersEquip(item, -newNumber);
		}
		$gameMap.requestRefresh();
	}
};

if(Imported.YEP_ItemCore){
	Game_Party.prototype.registerNewItem = function(baseItem, newItem) {
		var container = this.itemContainer(baseItem);
		var amount = 1;
		var item = newItem;
		if (container) {
			var lastNumber = this.numItems(item);
			var newNumber = lastNumber + amount;
			container[item.id] = newNumber;
			this._grid.pushItem(item, amount);
			// Clean up
			if (container[item.id] === 0) {
				delete container[item.id];
			}
			$gameMap.requestRefresh();
		}
	}

	Game_Party.prototype.removeIndependentItem = function(item, includeEquip) {
		if (includeEquip && this.checkItemIsEquipped(item)) {
		  for (var i = 1; i < $gameActors._data.length; ++i) {
			var actor = $gameActors.actor(i);
			if (!actor) continue;
			if (!actor.equips().contains(item)) continue;
			actor.unequipItem(item);
		  }
		}

		var container = this.itemContainer(item);
		var amount = -1;
		if (container) {
			var lastNumber = this.numItems(item);
			var newNumber = lastNumber + amount;
			container[item.id] = newNumber;
			this._grid.pushItem(item, amount);

			// Clean up
			if (container[item.id] === 0) {
				delete container[item.id];
			}
			$gameMap.requestRefresh();
		}
	};
}

Game_Party.prototype.numItems = function(item) {
	return this._grid ? this._grid.numItems(item) : 0;
};

Game_Actor.prototype.heldChangeEquip = function(slotId, item) {
	if (!item || this.equipSlots()[slotId] === item.etypeId) {
		$gameParty.gainItem(this.equips()[slotId], 1);
		this._equips[slotId].setObject(item);
		this.refresh();
	}
};

//-----------------------------------------------------------------------------
// Window
//
// Changes to Window

MRP.GridInventory.Window_initialize = Window.prototype.initialize;
Window.prototype.initialize = function() {
	MRP.GridInventory.Window_initialize.call(this);
	this._highlightWindowskin = null;
	this._baseWindowskin = null;
}

/**
 * Rather or not the highlight window skin should be used.
 *
 * @property highlight
 * @type Boolean
 */
Object.defineProperty(Window.prototype, 'highlight', {
    set: function(value) {
		var oldSkin = this._windowskin;
		this._windowskin = value ? this._highlightWindowskin : this._baseWindowskin;
		if(oldSkin != this._windowskin) this._onWindowskinLoad.call(this);
	},
	get: function() {
        return this._windowskin === this._highlightWindowskin && this._windowskin != null;
    },
    configurable: true
});

Object.defineProperty(Window.prototype, 'windowskin', {
	set: function(value) {
        if (this._baseWindowskin !== value) {
			this._windowskin = value;
            this._baseWindowskin = value;
            this._baseWindowskin.addLoadListener(this._onWindowskinLoad.bind(this));
        }
    },
});

/**
 * The image used as a window skin.
 *
 * @property highlightWindowskin
 * @type Bitmap
 */
Object.defineProperty(Window.prototype, 'highlightWindowskin', {
    set: function(value) {
        if (this._highlightWindowskin !== value) {
            this._highlightWindowskin = value;
            this._highlightWindowskin.addLoadListener(this._onWindowskinLoad.bind(this));
        }
    },
    configurable: true
});

//-----------------------------------------------------------------------------
// Window_Base
//
// Changes to Window_Base

MRP.GridInventory.Window_Base_loadWindowskin = Window_Base.prototype.loadWindowskin;
Window_Base.prototype.loadWindowskin = function() {
	MRP.GridInventory.Window_Base_loadWindowskin.call(this);
    this.highlightWindowskin = ImageManager.loadSystem('WindowHighlight');
};

//-----------------------------------------------------------------------------
// Window_ItemGrid
//
// The window for selecting an item on the item screen.
function Window_ItemGrid() {
	this.initialize.apply(this, arguments);
}

Window_ItemGrid.prototype = Object.create(Window_Selectable.prototype);
Window_ItemGrid.prototype.constructor = Window_ItemGrid;

Window_ItemGrid.prototype.initialize = function(x, y, width, height) {
	this._x = x;
	this._y = y;
	Window_Selectable.prototype.initialize.call(this, x, y, width, height);
	this._data = new Game_Grid("Inventory", 0, 0);
	this._cols = 0;
	this._rows = 0;
	this._gridHeight = MRP.GridInventory.cellSize;
	this._gridWidth = MRP.GridInventory.cellSize;
	this._toolTipWindow = null;
	this._playerInventory = false;
	this._activeMouse = true;
	this._activeKeyboard = true;
	
	// 0: default, 1: can do, 2: cannot do, 3: swap
	this._colorState = 0;
};

Window_ItemGrid.prototype.standardPadding = function() {
	return 9;
};

Window_ItemGrid.prototype.itemWidth = function() {
	return this._gridWidth;
};

Window_ItemGrid.prototype.itemHeight = function() {
	return this._gridHeight;
};
Window_ItemGrid.prototype.isMouseInsideFrame = function() {
	var x = this.canvasToLocalX(TouchInput._mouseX);
	var y = this.canvasToLocalY(TouchInput._mouseY);
	return x >= 0 && y >= 0 && x < this.width && y < this.height;
};
Window_ItemGrid.prototype.isOkTriggered = function() {
    return Input.isRepeated('ok') && this.isKeyboardActive();
};
Window_ItemGrid.prototype.update = function(){
	Window_Selectable.prototype.update.call(this);
	if(this.isOpenAndActive()){
		if(!this.gridItem().isEmpty()){
			// Display Tooltip
			if(!HeldItemManager.isHoldingItem()){
				var hoverItem = this.gridItem();
				this._toolTipWindow.setGridItem(hoverItem);
				var rect = this.itemRect(hoverItem._index, hoverItem);
				var x =  this._x + this.standardPadding() + rect.width/2 + rect.x - this._toolTipWindow.width/2;
				var y =  this._y + this.standardPadding() + rect.y;
				y += y <  Graphics.boxHeight / 2 ? rect.height : -this._toolTipWindow.height;
			
				x = x.clamp(0, Graphics.boxWidth - this._toolTipWindow.width);
				y = y.clamp(0, Graphics.boxHeight - this._toolTipWindow.height);
				this._toolTipWindow.setPosition(x, y, rect);
				this._toolTipWindow.show();
			}
		}
		if(this.isMouseActive()){
			// Update index
			if(TouchInput._mouseMoved) this.updateIndexMouse();
			if(this.isMouseInsideFrame()){
				// Redraw index if held item.
				if(this.index() != -1 && HeldItemManager.isHoldingItem()){
					var heldItem = HeldItemManager.heldItem();
					var pos = this._data.pos(this.index())
					for(var i = 0; i < heldItem._width; i++){
						for(var j = 0; j < heldItem._height; j++){
							var index = this._data.index(pos.col + i, pos.row + j);
							if(index < this.maxItems()) this.redrawItem(index, true);
						}
					}
				}
			}
		}
	}
}

Window_ItemGrid.prototype.setColorState = function(state){
	this._colorState = state;
}
Window_ItemGrid.prototype.getCursorColor = function(){
	switch(this._colorState){
		case 0:
			return MRP.GridInventory.selectColor;
		case 1:
			return MRP.GridInventory.usableColor;
		case 2:
			return MRP.GridInventory.unusableColor;
		case 3:
			return MRP.GridInventory.swapColor;
		default:
			return "#FFFFFF";
	}
}

Window_ItemGrid.prototype.setToolTipWindow = function(window) {
	this._toolTipWindow = window;
}

Window_ItemGrid.prototype.setGridSize = function(cols, rows){
	this._cols = cols;
	this._rows = rows;
}
Window_ItemGrid.prototype.bestMatchIndex = function(gridItem, x, y) {
	var width = gridItem._width;
	var height = gridItem._height;
	
	var cx = x - this.padding;
	var cy = y - this.padding;
	var topIndex = this.topIndex();
	// Find best potentional indices
	var potentional = [];
	for (var i = 0; i < this.maxPageItems(); i++) {
		var index = topIndex + i;
		if (index < this.maxItems()) {
			var rect = this.itemRect(index, gridItem);
			var right = rect.x + rect.width;
			var bottom = rect.y + rect.height;
			if (cx >= rect.x && cy >= rect.y && cx < right && cy < bottom) {
				var pos = this._data.pos(index);
				if(this._data.isWithinGrid(pos.col + width - 1, pos.row + height - 1)){
					var cRectX = rect.x + rect.width/2;
					var cRectY = rect.y + rect.height/2;
					var distance = Math.pow(cRectX - cx, 2) + Math.pow(cRectY - cy, 2);
					potentional.push({index: index, dist: distance});
				}
			}
		}
	}
	// Return the best potentional index
	if(potentional.length > 0){
		var best = potentional[0].index;
		var bestDist = potentional[0].dist;
		for(var i = 1; i < potentional.length; i++){
			if(potentional[i].dist < bestDist) {
				best = potentional[i].index;
				bestDist = potentional[i].dist;
			}
		}
		return best;
	} else {
		return -1;
	}
}
Window_ItemGrid.prototype.processCancel = function() {
	if(HeldItemManager.isHoldingItem()){
		var heldItem = HeldItemManager.heldItem();
		this._data.pushItem(heldItem.object(), heldItem._amount);
		HeldItemManager.setHeldItem(null);
	}
	Window_Selectable.prototype.processCancel.call(this);
};

Window_ItemGrid.prototype.itemRect = function(index, gridItem) {
	gridItem = gridItem ? gridItem : this.gridItem(index);
	var rect = new Rectangle();
	var maxCols = this.maxCols();
	rect.width = this.itemWidth();
	rect.height = this.itemHeight();
	var cY = (this.contents.height + this.lineHeight() - rect.height * this._rows) / 2;
	var cX = (this.contents.width - rect.width * this._cols) / 2;;
	rect.x = cX + index % maxCols * (rect.width + this.spacing()) - this._scrollX;
	rect.y = cY + Math.floor(index / maxCols) * (rect.height + this.spacing()) - this._scrollY;
	
	rect.width *= gridItem._width;
	rect.height *= gridItem._height;
	return rect;
};

Window_ItemGrid.prototype.spacing = function() {
	return 0;
};

Window_ItemGrid.prototype.maxCols = function() {
	return this._cols;
};

Window_ItemGrid.prototype.maxRows = function() {
	return this._rows;
};

Window_ItemGrid.prototype.maxItems = function() {
	return this.maxCols() * this.maxRows();
};

Window_ItemGrid.prototype.item = function() {
	return this.gridItem() ? this.gridItem().object() : null;
};

Window_ItemGrid.prototype.gridItem = function(index) {
	var index = index == undefined ? this.index() : index;
	if(this._data && index >= 0 && index < this.maxItems()){
		var pos = this._data.pos(index);
		var gridItem = this._data._grid[pos.col][pos.row];
		if(gridItem._parent != null) {
			var parentPos = this._data.pos(gridItem._parent);
			gridItem = this._data._grid[parentPos.col][parentPos.row];
		}
		return gridItem;
	}
	return this._data ? this._data.createEmptyItem(-1) : null;
};
Window_ItemGrid.prototype.setPlayerInventory = function(bool) {
	this._playerInventory = bool;
	this.refresh();
};
Window_ItemGrid.prototype.isPlayerInventory = function() {
	return this._playerInventory;
};
Window_ItemGrid.prototype.setData = function(data) {
	this._data = data;
	this.setGridSize(data._cols, data._rows);
	if(!this.isPlayerInventory()) this.refresh();
};

Window_ItemGrid.prototype.updateItemList = function() {
	if(this.isPlayerInventory()){
		var items = $gameParty.gridItems();
		this.setData(items);
	}
};

Window_ItemGrid.prototype.selectLast = function() {
	this.select(0);
};

Window_ItemGrid.prototype.redrawItem = function(index, select) {
	index = this.gridItem(index)._index;
	if (index >= 0) {
		this.clearItem(index);
		this.drawItem(index, select);
	}
};

Window_ItemGrid.prototype.drawItem = function(index, select) {
	var col = index % this._cols;
	var row = Math.floor(index / this._cols);
	var item = this._data._grid[col][row];
	if(item._parent == null) {
		this.drawGridLines(index);
		if (!item.isEmpty()) {
			var rect = this.itemRect(index);
			if(select) this.drawSelectBackground(item);
			else this.drawBackground(index, item, rect);
			rect.width -= this.textPadding();
			this.drawItemImage(item, rect.x, rect.y, rect.width);
			this.makeFontSmaller();
			this.drawItemNumber(item, rect.x + this.textPadding(), rect.y - this.textPadding(), rect.width);
			this.resetFontSettings();
		}			
	}
};
Window_ItemGrid.prototype.drawSelectBackground = function(gridItem){
	if(HeldItemManager.isHoldingItem()){
		this.contents.paintOpacity =  this._windowCursorSprite.alpha * 255;			
		var rect = this.itemRect(gridItem._index, gridItem);
		this.contents.fillRect(rect.x, rect.y, rect.width, rect.height, this.getCursorColor());
		this.contents.paintOpacity = 255;
	}
}
Window_ItemGrid.prototype.drawBackground = function(index, item, rect) {
	this.changePaintOpacity(false);
	var colorCode = this.getBackgroundColor(item);
	
	this.contents.fillRect(rect.x, rect.y, rect.width, rect.height, colorCode);
	this.changePaintOpacity(true);
}

Window_ItemGrid.prototype.getBackgroundColor = function(item){
	var colorCode;
	if(MRP.GridInventory.useItemBackGroundRarity){
		var rarity = item.object().meta.Rarity;
		if(rarity) colorCode = MRP.GridInventory.rarities[rarity].Color;
		else colorCode = MRP.GridInventory.defaultRarity.Color;
	} else {
		if(item.isEnabled()) {
			colorCode = MRP.GridInventory.usableBackgroundColor;
		} else {
			colorCode = MRP.GridInventory.unusableBackgroundColor;
		}	
	}
	return colorCode;
}

Window_ItemGrid.prototype.drawItemImage = function(item, x, y, width) {
	width = width || 312;
	if (item) {
		this.resetTextColor();
		this.drawIcon(item, x, y, item._width * (MRP.GridInventory.cellSize), item._height * (MRP.GridInventory.cellSize));
	}
};
Window_ItemGrid.prototype.drawIcon = function(item, x, y, width, height) {
	// Draw Custom Icon
	if(item.object().meta.iconImage){
		var bitmap = ImageManager.loadSystem('InventoryIcons');
		var iconInfo = $dataInventoryIcons.frames[item.object().meta.iconImage]; 
		var pw = iconInfo.frame.w;
		var ph = iconInfo.frame.h;
		var sx = iconInfo.frame.x;
		var sy = iconInfo.frame.y;
		var ww = width;
		var wh = height;
		// The size it should be drawn with
		var dw = pw;
		var dh = ph;

		// Only ever scales down. Never up.
		if(ww < dw) {
			dh = dh * ww / dw;
			dw = ww;
		}

		if(wh < dh){
			dw = dw * wh / dh;
			dh = wh;
		}

		// Center Image
		y += (wh - dh) / 2;
		x += (ww - dw) / 2;

		
		this.contents.blt(bitmap, sx, sy, pw, ph, x, y, dw, dh);
	// Draw Regular Icon
	} else {
		var bitmap = ImageManager.loadSystem('IconSet');
		var iconIndex = item.object().iconIndex;
		var pw = Window_Base._iconWidth;
		var ph = Window_Base._iconHeight;
		var sx = iconIndex % 16 * pw;
		var sy = Math.floor(iconIndex / 16) * ph;
		// The size it should be drawn with
		var dw = width;
		var dh = height;

		this.contents.blt(bitmap, sx, sy, pw, ph, x, y, dw, dh);
	}
	
};

Window_ItemGrid.prototype.drawItemNumber = function(item, x, y, width) {
	if (this.needsNumber(item)) {
		this.drawText(item.numItems(), x, y, width, 'left');
	}
};

Window_ItemGrid.prototype.needsNumber = function(item) {
	return item._amount > 1;
};

Window_ItemGrid.prototype.refresh = function() {
	this.updateItemList();
	this.createContents();
	this.drawAllItems();
	this.updateCursor();
};

Window_ItemGrid.prototype.drawAllItems = function(){
	// Draw Title
	this.drawText(this._data._name, 0, 0, this.contentsWidth(), 'center');
	// Draw Items
	Window_Selectable.prototype.drawAllItems.call(this);
}


Window_ItemGrid.prototype.drawGridLines = function(index) {
	var rect = this.itemRect(index);
	var x1 = rect.x;
	var y1 = rect.y;
	var x2 = x1 + rect.width - 1;
	var y2 = y1 + rect.height - 1;
	
	this.drawHorzLine(x1, y1);
	this.drawHorzLine(x1, y2);
	this.drawVertLine(x1, y1);
	this.drawVertLine(x2, y1);
};

Window_ItemGrid.prototype.drawHorzLine = function(x, y) {
	var lineY = y;
	this.contents.paintOpacity = 48;
	this.contents.fillRect(x, lineY, this.itemWidth(), 1, this.lineColor());
	this.contents.paintOpacity = 255;
};

Window_ItemGrid.prototype.drawVertLine = function(x, y) {
	this.contents.paintOpacity = 48;
	this.contents.fillRect(x, y, 1, this.itemHeight(), this.lineColor());
	this.contents.paintOpacity = 255;
};

Window_ItemGrid.prototype.lineColor = function() {
	return this.normalColor();
};

Window_ItemGrid.prototype.isCurrentItemEnabled = function() {
	return this.index() > -1 && (this.isEnabled(this.gridItem()) || HeldItemManager.isHoldingItem());
};

Window_ItemGrid.prototype.isEnabled = function(gridItem) {
	return !gridItem.isEmpty();
};
Window_ItemGrid.prototype.isMouseActive = function() {
	return this._activeMouse == true;
};
Window_ItemGrid.prototype.isKeyboardActive = function() {
	return this._activeKeyboard == true;
};
Window_ItemGrid.prototype.activateMouse = function() {
	this._activeMouse = true;
};
Window_ItemGrid.prototype.deactivateMouse = function() {
	this._activeMouse = false;
};
Window_ItemGrid.prototype.activateKeyboard = function() {
	this._activeKeyboard = true;
	this.highlight = true;
};
Window_ItemGrid.prototype.deactivateKeyboard = function() {
	this._activeKeyboard = false;
	this.highlight = false;
};

Window_ItemGrid.prototype.dropItem = function(){
	var grid = this._data;
	var pos = grid.pos(this.index());
	grid.removeItem(pos.col, pos.row);
	this.refresh();
}
Window_ItemGrid.prototype.processTouch = function() {
	if (this.isOpenAndActive() && this.isMouseActive() && this.isMouseInsideFrame()) {
		// Left Click Inside Frame
		if (TouchInput.isTriggered()) {
			this._touching = true;
			this.onTouch(true);
		// Right Click
		} else if (TouchInput.isCancelled()) {
			if(HeldItemManager.isHoldingItem()){
				var pos = this._data.pos(this.index());
				if(this._data.canPlaceHeldItem(HeldItemManager.heldItem(), pos.col, pos.row)){
					this.processOk();
				} else {
					SoundManager.playBuzzer();
				}
			} else {
				var x = this.canvasToLocalX(TouchInput.x);
				var y = this.canvasToLocalY(TouchInput.y);
				var hitIndex = this.hitTest(x, y);
				if (hitIndex >= 0 && !this.gridItem(hitIndex).isEmpty() && this.isTouchOkEnabled()) {
					this.select(this.gridItem(hitIndex)._index);
					this.processOk();
				} else {
					this.processCancel();
				}
			}
		}
		if (this._touching) {
			if (TouchInput.isPressed()) {
				this.onTouch(false);
			} else {
				this._touching = false;
			}
		}
	} else {
		this._touching = false;
	}
};
Window_ItemGrid.prototype.processOk = function() {
	if (this.isCurrentItemEnabled()) {
		this.playOkSound();
		this.callOkHandler();
		this.updateInputData();		
	}
};

Window_ItemGrid.prototype.ensureCursorVisible = function() {
	Window_Selectable.prototype.ensureCursorVisible.call(this);
	if(this._data && HeldItemManager.isHoldingItem()){
		var heldItem = HeldItemManager.heldItem();
		var pos = this._data.pos(this.index());
		for(var i = 0; i < heldItem._width; i++){
			if(!this._data.isWithinGrid(pos.col + i, pos.row)){
				this._index -= heldItem._width - i;
				break;
			}
		}
		for(var j = 0; j < heldItem._height; j++){
			if(!this._data.isWithinGrid(pos.col, pos.row + j)){
				this._index -= (heldItem._height - j) * this._cols;
				break;
			}
		}
	}
};

Window_ItemGrid.prototype.select = function(index) {
	if(index >= this.maxItems()) return;
	var prevItem = this.gridItem(this.index());
	var gridItem = this.gridItem(index);
	if(gridItem != null && !HeldItemManager.isHoldingItem()){
		if(!(TouchInput._mouseMoved || TouchInput.isTriggered()) && prevItem && gridItem == prevItem && index != this.index()) {
			index - this.index() == 1 ? this.select(index + 1): this.select(index + this._cols);
			return;
		} else {
			this._index = gridItem._children.length > 0 ? gridItem._index : index;
		}
	} else {
		this._index = index;
	}
	if(prevItem && !(prevItem._children.includes(this.index()) || prevItem._index == this.index())) 
	{ 
		this.refresh();
	}
	this._stayCount = 0;
	this.ensureCursorVisible();
	this.updateCursor();
	this.callUpdateHelp();
	this._index = Math.max(-1, this._index);
};
Window_ItemGrid.prototype.getRectPos = function() {
	var pad = this._padding;
	var x1 = this._cursorRect.x + pad + 3;
	var y1 = this._cursorRect.y + pad + 3;
	var ox = x1 + this.x;
	var oy = y1 + this.y;
	return {x: ox, y: oy};
}
Window_ItemGrid.prototype.updateIndexMouse = function() {
	var x = this.canvasToLocalX(TouchInput._mouseX);
	var y = this.canvasToLocalY(TouchInput._mouseY);
	if (this.isContentsArea(x, y)) {
		if(HeldItemManager.isHoldingItem()) {
			var heldItem = HeldItemManager.heldItem();
			var bestIndex = this.bestMatchIndex(heldItem, x, y);
			this.select(bestIndex);
		} else {
			var index = this.hitTest(x, y);
			this.select(index);
		}
	} else {
		this.select(-1);
	}
}

Window_ItemGrid.prototype.processCursorMove = function() {
    if (this.isKeyboardActive()) {
        Window_Selectable.prototype.processCursorMove.call(this);
    }
};

Window_ItemGrid.prototype.updateCursor = function() {
	if(this.isCursorVisible() && HeldItemManager.isHoldingItem()) {
		var heldItem = HeldItemManager.heldItem();
		var rect = this.itemRect(this.index(), heldItem);
		
		var pos = this._data.pos(this.index())
		var canPlaceItem = this._data.canPlaceHeldItem(heldItem, pos.col, pos.row);
		if(canPlaceItem) {
			if(canPlaceItem !== true) this.setColorState(3);
			else this.setColorState(1);
		}
		else {
			this.setColorState(2);
		}
		
		this.setCursorRect(rect.x, rect.y, rect.width, rect.height);
		if(!TouchInput._mouseMoved){
			var pos = this.getRectPos();
			HeldItemManager.setPosition(pos.x, pos.y);
		}
	} else {
		this.setColorState(0);
		if (this._cursorAll) {
			var allRowsHeight = this.maxRows() * this.itemHeight();
			this.setCursorRect(0, 0, this.contents.width, allRowsHeight);
			this.setTopRow(0);
		} else if (this.isCursorVisible()) {
			if(!this._data.gridItem(this.index()).isEmpty()) this.setColorState(1);
			var rect = this.itemRect(this.index());
			this.setCursorRect(rect.x, rect.y, rect.width, rect.height);
		} else {
			this.setCursorRect(0, 0, 0, 0);
		}
	}
};
Window_ItemGrid.prototype.hitTest = function(x, y) {
	if (this.isContentsArea(x, y)) {
		var cx = x - this.padding;
		var cy = y - this.padding;
		var topIndex = this.topIndex();
		for (var i = 0; i < this.maxPageItems(); i++) {
			var index = topIndex + i;
			if (index < this.maxItems()) {
				var rect = this.itemRect(index);
				var right = rect.x + this._gridWidth;
				var bottom = rect.y + this._gridHeight;
				if (cx >= rect.x && cy >= rect.y && cx < right && cy < bottom) {
					return index;
				}
			}
		}
	}
	return -1;
};
Window_ItemGrid.prototype.onTouch = function(triggered) {
	var lastIndex = this.index();
	var x = this.canvasToLocalX(TouchInput.x);
	var y = this.canvasToLocalY(TouchInput.y);
	var hitIndex = this.hitTest(x, y);
	if (hitIndex >= 0) {
		if (hitIndex === this.index()) {
			if (triggered && this.isTouchOkEnabled()) {
				this.processOk();
			}
		} else if (this.gridItem(hitIndex) || HeldItemManager.isHoldingItem()){
			if (triggered && this.isTouchOkEnabled()) {
				if(!HeldItemManager.isHoldingItem()) this.select(hitIndex);
				this.processOk();
			}
		} else if (this.isCursorMovable()) {
			this.select(hitIndex);
		}
	}
	
	if (this.index() !== lastIndex) {
		SoundManager.playCursor();
	}
};

Window_ItemGrid.prototype._refreshCursor = function() {
	var pad = this._padding;
	var x = this._cursorRect.x + pad - this.origin.x;
	var y = this._cursorRect.y + pad - this.origin.y;
	var w = this._cursorRect.width;
	var h = this._cursorRect.height;
	var x2 = Math.max(x, pad);
	var y2 = Math.max(y, pad);
	var w2 = Math.min(w, this._width - pad - x2);
	var h2 = Math.min(h, this._height - pad - y2);
	var bitmap = new Bitmap(w2, h2);

	this._windowCursorSprite.bitmap = bitmap;
	this._windowCursorSprite.setFrame(0, 0, w2, h2);
	this._windowCursorSprite.move(x2, y2);
	if (w > 0 && h > 0) {
		bitmap.fillRect(0, 0, w2, h2, this.getCursorColor());
	}
};

//-----------------------------------------------------------------------------
// Window_Equip
//
// The window class of the equip item window.

function Window_Equip() {
	this.initialize.apply(this, arguments);
}
Window_Equip.prototype = Object.create(Window_Base.prototype);
Window_Equip.prototype.constructor = Window_Equip;
Window_Equip.prototype.initialize = function(x, y, width, height) {
	Window_Base.prototype.initialize.call(this, x, y, width, height);
	this._handlers = {};
	this._actor = null;
	this._equipSlots = [];
	this._toolTipWindow = null;
	this._characterSprite = new Sprite(0, 0);
};

Window_Equip.prototype.setHandler = function(symbol, method) {
	this._handlers[symbol] = method;
};
Window_Equip.prototype.isHandled = function(symbol) {
	return !!this._handlers[symbol];
};

Window_Equip.prototype.callHandler = function(symbol) {
	if (this.isHandled(symbol)) {
		this._handlers[symbol]();
	}
};

Window_Equip.prototype.setHighlight = function(value){
	this.highlight = value;
}

Window_Equip.prototype.update = function(){
	Window_Base.prototype.update.call(this);
	if(TouchInput.isCancelled() && this.isTouchedInsideFrame() && !HeldItemManager.isHoldingItem()){
		this.callCancelHandler();
	}
}

Window_Equip.prototype.refresh = function(){
	this.contents.clear();
	var equipType = this.getEquipType();
	this.evalCustomCode(equipType);
}

Window_Equip.prototype.updateInputData = function() {
    Input.update();
    TouchInput.update();
};

Window_Equip.prototype.callCancelHandler = function() {
	SoundManager.playCancel();
	this.updateInputData();
	this.callHandler('cancel');
};

Window_Equip.prototype.resetArrowVisibility = function(){
	this.leftArrowVisible = false;
	this.rightArrowVisible = false;
}

Window_Equip.prototype.showLeftArrow = function(bool){
	this.leftArrowVisible = bool;
}

Window_Equip.prototype.showRightArrow = function(bool){
	this.rightArrowVisible = bool;
}

Window_Equip.prototype._createAllParts = function() {
	Window.prototype._createAllParts.call(this);

    this._leftArrowSprite = new Sprite();
    this._rightArrowSprite = new Sprite();
    this.addChild(this._leftArrowSprite);
    this.addChild(this._rightArrowSprite);

};

Window_Equip.prototype._refreshArrows = function() {
	Window.prototype._refreshArrows.call(this);
    var w = this._width;
    var h = this._height;
    var p = 24;
    var q = p/2;
    var sx = 96+p;
    var sy = 0+p;
    this._rightArrowSprite.bitmap = this._windowskin;
    this._rightArrowSprite.anchor.x = 0.5;
    this._rightArrowSprite.anchor.y = 0.5;
    this._rightArrowSprite.setFrame(sx + p + q, sy+q, q, p);
    this._rightArrowSprite.move(w-q, h/2);
    this._leftArrowSprite.bitmap = this._windowskin;
    this._leftArrowSprite.anchor.x = 0.5;
    this._leftArrowSprite.anchor.y = 0.5;
    this._leftArrowSprite.setFrame(sx, sy+q, q, p);
    this._leftArrowSprite.move(q, h/2);
};

Window_Equip.prototype._updateArrows = function() {
	Window.prototype._updateArrows.call(this);
    this._rightArrowSprite.visible = this.isOpen() && this.rightArrowVisible;
    this._leftArrowSprite.visible = this.isOpen() && this.leftArrowVisible;
};


Window_Equip.prototype.getEquipType = function() {
	var type = $dataClasses[this._actor._classId].meta.equipType;
	if(type != undefined) {
		for(var i = 0; i < MRP.GridInventory.equipWindowTypes.length; i++){
			var name = MRP.GridInventory.equipWindowTypes[i].name;
			if(name == type) return MRP.GridInventory.equipWindowTypes[i];
		}
	}
	return MRP.GridInventory.equipWindowTypes[0];
}

Window_Equip.prototype.drawVertLine = function(x) {
	this.contents.paintOpacity = 48;
	this.contents.fillRect(x, 0, 1, this.contentsHeight(), this.normalColor());
	this.contents.paintOpacity = 255;
};

Window_Equip.prototype.addEquipSlots = function() {
	var equipType = this.getEquipType();
	var equipSlots = equipType.equipSlots;
	var cell = MRP.GridInventory.cellSize;

	for(var i = 0; i < equipSlots.length; i++){
		var eType = this._actor.equipSlots()[i];
		if(eType == undefined) break;

		var slot = equipSlots[i];
		this.addEquipSlot(i, eval(slot.x), eval(slot.y), eval(slot.width), eval(slot.height));
	}

	this.refresh();

}

Window_Equip.prototype.evalCustomCode = function(equipType) {
	var a = this._actor;
	var w = this.contentsWidth();
	var h = this.contentsHeight();
	var cell = MRP.GridInventory.cellSize;

	eval(equipType.customCode);
}

Window_Equip.prototype.addEquipSlot = function(slot, x, y, width, height) {
	var equipSlot = new Window_EquipGridSlot(x, y, width, height);
	equipSlot.setHandler('onEquip', this.onEquip.bind(this));
	equipSlot.setGridSlot(slot);
	equipSlot.activate();
	this._equipSlots.push(equipSlot);
	this.addChildToBack(equipSlot);
}

Window_Equip.prototype.getEquipSlotIDs = function(etypeId, startIndex, endIndex){
	if(startIndex === undefined) startIndex = 0;
	if(endIndex === undefined) endIndex = this._equipSlots.length;
	var slots = [];
	for(var i = startIndex; i < endIndex; i++){
		var equipSlot = this._equipSlots[i];
		if(this._actor.equipSlots()[equipSlot._gridSlot] == etypeId) slots.push(i);
	}
	return slots;
}

Window_Equip.prototype.activate = function() {
	Window_Base.prototype.activate.call(this);
	this.activateSlotWindows();
};

Window_Equip.prototype.deactivate = function() {
	Window_Base.prototype.deactivate.call(this);
	this.deactivateSlotWindows();
};

Window_Equip.prototype.activateSlotWindows = function() {
	for(var i = 0; i < this._equipSlots.length; i++){
		this._equipSlots[i].activate();
	}
};

Window_Equip.prototype.deactivateSlotWindows = function() {
	for(var i = 0; i < this._equipSlots.length; i++){
		this._equipSlots[i].deactivate();
	}
};


Window_Equip.prototype.setToolTipWindow = function(window) {
	this._toolTipWindow = window;
	for(var i = 0; i < this._equipSlots.length; i++){
		this._equipSlots[i].setToolTipWindow(this._toolTipWindow);
	}
}

Window_Equip.prototype.isTouchedInsideFrame = function() {
	var x = this.canvasToLocalX(TouchInput.x);
	var y = this.canvasToLocalY(TouchInput.y);
	return x >= 0 && y >= 0 && x < this.width && y < this.height;
};
Window_Equip.prototype.setActor = function(actor) {
	this._actor = actor;
	this._characterSprite = new Sprite_Character();
	this.addEquipSlots();
	for(var i = 0; i < this._equipSlots.length; i++){
		this._equipSlots[i].setActor(actor);
	}
};
Window_Equip.prototype.onEquip = function() {
	this.callHandler('onEquip');
	this.refresh();
};
Window_Equip.prototype.updateGridItems = function(){
	for(var i = 0; i < this._equipSlots.length; i++){
		this._equipSlots[i].updateGridItem();
	}
}

//-----------------------------------------------------------------------------
// Window_EquipGridSlot
//
// The scene class of the grid item screen.

function Window_EquipGridSlot() {
	this.initialize.apply(this, arguments);
}

Window_EquipGridSlot.prototype = Object.create(Window_Selectable.prototype);
Window_EquipGridSlot.prototype.constructor = Window_EquipGridSlot;

Window_EquipGridSlot.prototype.initialize = function(x, y, width, height) {
	Window_Selectable.prototype.initialize.call(this, x, y, width, height);
	this._gridSlot = -1;
	this._gridItem = null;
	this._actor = null;
	this._toolTipWindow = null;
	this._backgroundColor = "#FFFFFF";
	this.select(-1);
};

Window_EquipGridSlot.prototype.setToolTipWindow = function(window) {
	this._toolTipWindow = window;
}

Window_EquipGridSlot.prototype.processOk = function() {
	if (this.isCurrentItemEnabled()) {
		if(HeldItemManager.isHoldingItem()){
			this.equipHeldItem();
		} else if(this._gridItem) {
			this.unEquipItem();				
		}
		this.updateInputData();
	}
};
Window_EquipGridSlot.prototype.equipHeldItem = function() {
	var heldItem = HeldItemManager.heldItem().object();
	var eTypeId = this._actor.equipSlots()[this._gridSlot];
	if(eTypeId == heldItem.etypeId && this._actor.canEquip(heldItem)){	
		if(this._gridItem){				
			this._actor.forceChangeEquip(this._gridSlot, heldItem);
			HeldItemManager.setHeldItem(this._gridItem);
		} else {
			this._actor.heldChangeEquip(this._gridSlot, heldItem);
			HeldItemManager.setHeldItem(null);
		}
		this.callHandler('onEquip');
		this.setHighlight(false);
		SoundManager.playEquip();
		this.updateGridItem();
		return true;
	} else {
		SoundManager.playBuzzer();
		return false;
	}
};
Window_EquipGridSlot.prototype.unEquipItem = function() {
	HeldItemManager.setHeldItem(this._gridItem);
	this._actor.forceChangeEquip(this._gridSlot, null);
	SoundManager.playOk();
	this.callHandler('onEquip');
	this.updateGridItem();
}
Window_EquipGridSlot.prototype.onTouch = function(triggered) {
	var x = this.canvasToLocalX(TouchInput.x);
	var y = this.canvasToLocalY(TouchInput.y);
	var hitIndex = this.hitTest(x, y);
	if (hitIndex >= 0) {
		if (triggered) {
			this.processOk();
		}
	}
};
Window_EquipGridSlot.prototype.update = function() {
	Window_Selectable.prototype.update.call(this);
	if(this.isOpenAndActive()){
		var highlight = false;
		if(this._gridItem && (Window_ItemGrid.prototype.isMouseInsideFrame.call(this) || this.highlight)){
			var hoverItem = this._gridItem;
			this._toolTipWindow.setGridItem(hoverItem);
			var rect = this.itemRect(0);
			var x =  this.parent.x + this.x + this.standardPadding() + rect.width/2 + rect.x - this._toolTipWindow.width/2;
			var y =  this.parent.y + this.y + this.standardPadding() + rect.y;
	
			y += y <  Graphics.boxHeight / 2 ? rect.height : -this._toolTipWindow.height;
			
			x = x.clamp(0, Graphics.boxWidth - this._toolTipWindow.width);
			y = y.clamp(0, Graphics.boxHeight - this._toolTipWindow.height);
			
			this._toolTipWindow.setPosition(x, y, rect);
			this._toolTipWindow.show();
		}
	
		if(HeldItemManager.isHoldingItem()){
			if(!HeldItemManager.heldItem().isItem()){
				var heldItem = HeldItemManager.heldItem().object();
				var etypeId = this._actor.equipSlots()[this._gridSlot];
				if(etypeId == heldItem.etypeId){
					highlight = true
				}
			}
		}
		if(highlight){
			if(this._actor.canEquip(HeldItemManager.heldItem().object())) this.setBackgroundColor("#58D3F7");
			else this.setBackgroundColor("#FF0000");
			if(this.index() != 0) this.select(0);
		} 
		else if(!highlight) this.select(-1);
	}
};

Window_EquipGridSlot.prototype.setHighlight = function(value){
	this.highlight = value;
}

Window_EquipGridSlot.prototype.setGridSlot = function(slot) {
	this._gridSlot = slot;
	this.updateGridItem();
}
Window_EquipGridSlot.prototype.setActor = function(actor) {
	this._actor = actor;
	this.updateGridItem();
}
Window_EquipGridSlot.prototype.updateGridItem = function() {
	if(this._actor != null && this._gridSlot != null) {
		var item = this._actor.equips()[this._gridSlot];
		this._gridItem = item ? new Game_GridItem(0, item, 1) : null;
		this.refresh();
	}
}
Window_EquipGridSlot.prototype.maxItems = function() {
	return 1;
};
Window_EquipGridSlot.prototype.isCursorMovable = function() {
	return false;
};
Window_EquipGridSlot.prototype.itemHeight = function() {
	return this.height - this._padding * 2;
};
Window_EquipGridSlot.prototype.standardPadding = function() {
	return 5;
};

Window_EquipGridSlot.prototype.drawItem = function(index) {
	var x = 0;
	var y = 0;
	var item = this._gridItem;
	if(!item || !item.object()) return;
	Window_ItemGrid.prototype.drawIcon.call(this, item, x, y, this.contentsWidth(), this.contentsHeight());
};
Window_EquipGridSlot.prototype._refreshCursor = function() {
	var pad = this._padding;
	var x = this._cursorRect.x + pad - this.origin.x;
	var y = this._cursorRect.y + pad - this.origin.y;
	var w = this._cursorRect.width;
	var h = this._cursorRect.height;
	var x2 = Math.max(x, pad);
	var y2 = Math.max(y, pad);
	var w2 = Math.min(w, this._width - pad - x2);
	var h2 = Math.min(h, this._height - pad - y2);
	var bitmap = new Bitmap(w2, h2);

	this._windowCursorSprite.bitmap = bitmap;
	this._windowCursorSprite.setFrame(0, 0, w2, h2);
	this._windowCursorSprite.move(x2, y2);
	if (w > 0 && h > 0) {
		bitmap.fillRect(0, 0, w2, h2, this.getBackgroundColor());
	}
};
Window_EquipGridSlot.prototype.setBackgroundColor = function(color){
	if(this._backgroundColor != color){
		this._backgroundColor = color;
		this._refreshCursor();
	}
}
Window_EquipGridSlot.prototype.getBackgroundColor = function(){
	return this._backgroundColor;
}

//-----------------------------------------------------------------------------
// Window_DynamicCommand
//
// The window class of the commands when selecting an item

function Window_DynamicCommand() {
	this.initialize.apply(this, arguments);
}
Window_DynamicCommand.prototype = Object.create(Window_Command.prototype);
Window_DynamicCommand.prototype.constructor = Window_DynamicCommand;
Window_DynamicCommand.prototype.initialize = function(commands, x, y) {
	this._useCommands = commands;
	Window_Command.prototype.initialize.call(this, x, y);
	this._item = null;
	this._widthNeeded = 0;
};

Window_DynamicCommand.prototype.setUseCommands = function(commands) {
	this._useCommands = commands;
}

Window_DynamicCommand.prototype.addUseCommand = function(command) {
	this._useCommands.push(command);
}

Window_DynamicCommand.prototype.removeUseCommand = function(command) {
	var pos = this._useCommands.indexOf(command);
	if(pos != -1) this._useCommands.splice(pos, 1);
}

Window_DynamicCommand.prototype.setItem = function(item) {
	this._item = item;
	this.refresh();
};

Window_DynamicCommand.prototype.refresh = function() {
	this._widthNeeded = 0;
	Window_Command.prototype.refresh.call(this);
	this.updateWidth();
};

Window_DynamicCommand.prototype.updateHeight = function() {
	this.height = this.windowHeight();
};

Window_DynamicCommand.prototype.updateWidth = function() {
	var newWidth = Math.max(this._widthNeeded + this.standardPadding() * 3, this.windowWidth());
	if(this.width != newWidth){
		this.width = newWidth;
		this.refresh();
	}
};

Window_DynamicCommand.prototype.drawItem = function(index) {
    var rect = this.itemRectForText(index);
    var align = this.itemTextAlign();
    this.resetTextColor();
    this.changePaintOpacity(this.isCommandEnabled(index));
	var widthNeeded = this.drawTextEx(this.commandName(index), rect.x, rect.y);
	if(this._widthNeeded < widthNeeded) {
		this._widthNeeded = widthNeeded;
	}
};

Window_DynamicCommand.prototype.numVisibleRows = function() {
	return this.maxItems();
};

Window_DynamicCommand.prototype.setPosition = function(x, y) {
	x = x.clamp(0, Graphics.boxWidth - this.width);
	y = y.clamp(0, Graphics.boxHeight - this.height);
	this.move(x, y, this.width, this.height);
};

//-----------------------------------------------------------------------------
// Window_ItemCommand
//
// The window class of the commands when selecting an item

Game_Party.prototype.canEquip = function(item){
	return this.members().some(function(actor) {
        return actor.canEquip(item);
    });
}

function Window_ItemCommand() {
	this.initialize.apply(this, arguments);
}
Window_ItemCommand.prototype = Object.create(Window_DynamicCommand.prototype);
Window_ItemCommand.prototype.constructor = Window_ItemCommand;
Window_ItemCommand.prototype.initialize = function(x, y) {
	Window_DynamicCommand.prototype.initialize.call(this, ['USE', 'MOVE', 'DROP', 'RENAME', 'AUGMENT'], x, y);
};

Window_ItemCommand.prototype.makeCommandList = function() {
	if(this._useCommands.includes('USE')) this.addCommand(DataManager.isItem(this._item) ? MRP.GridInventory.selectItemWindowConfig.textUseItem : MRP.GridInventory.selectItemWindowConfig.textEquipEquipment, 'use', DataManager.isItem(this._item) ? true : $gameParty.canEquip(this._item))
	if(this._useCommands.includes('MOVE')) this.addCommand(MRP.GridInventory.selectItemWindowConfig.textMove, 'move');
	if(this._useCommands.includes('DROP')) this.addCommand(MRP.GridInventory.selectItemWindowConfig.textDrop, 'drop');
	if(Imported.YEP_X_ItemRename){
		if(Window_ItemActionCommand.prototype.isShowRenameCommand.call(this)){
			var commandName = "Rename" //Window_ItemActionCommand.prototype.createRenameCommandName.call(this);
			if(this._useCommands.includes('RENAME')) this.addCommand(commandName, 'rename');
		}
	}
	if(Imported.YEP_X_AttachAugments){
		if(this._item && this._item.augmentSlots){
			ItemManager.checkAugmentSlots(this._item);
			for (var i = 0; i < this._item.augmentSlots.length; i++) {
				if(this._item.augmentSlotItems[i] != "none"){
					var enabled = this._item.augmentSlotEnable[i];
					var fmt = Yanfly.Param.AugmentSlotFmt;
					var slot = this._item.augmentSlots[i];
					var name = Window_ItemActionCommand.prototype.getAugmentSlotItemName.call(this, i);
					var text = Yanfly.Param.AugmentRemoveText + ": " + name; //fmt.format(slot, name);
					if(this._useCommands.includes('AUGMENT')) this.addCommand(text, 'augment', enabled, i);
				}
			}
			this.changePaintOpacity(true);
		}
	}
	if(this._item) this.updateHeight();
};


//-----------------------------------------------------------------------------
// Window_ItemHeldCommand
//
// The window class of the commands when selecting an item

function Window_ItemHeldCommand() {
	this.initialize.apply(this, arguments);
}
Window_ItemHeldCommand.prototype = Object.create(Window_DynamicCommand.prototype);
Window_ItemHeldCommand.prototype.constructor = Window_ItemHeldCommand;
Window_ItemHeldCommand.prototype.initialize = function(x, y) {
	Window_DynamicCommand.prototype.initialize.call(this, ['SWAP', 'DISASSEMBLE', 'REPAIR', 'UPGRADE', 'AUGMENT'], x, y);
	this._targetItem = null;
};

Window_ItemHeldCommand.prototype.makeCommandList = function() {
	if(this._useCommands.includes('SWAP')) this.addCommand(this._targetItem != null ? MRP.GridInventory.useItemWindowConfig.textSwap : MRP.GridInventory.useItemWindowConfig.textPlace, 'swap')
	// Support for YEP_X_ItemDisassemble
	if(this._targetItem != null && this._item != null){
		if(Imported.YEP_X_ItemDisassemble){
			if(ItemManager.isDisassembler(this._item)){
				if(this._useCommands.includes('DISASSEMBLE')) this.addCommand("Disassemble", 'disassemble');
			}
		}
		if(Imported.YEP_X_ItemDurability){
			if(Window_RepairItemList.prototype.containsType.call({"_item":this._targetItem}, this._item)){
				var enabled = Window_RepairItemList.prototype.isEnabled.call({"_item":this._targetItem}, this._item)
				if(this._useCommands.includes('REPAIR')) this.addCommand("Repair", 'repair', enabled);
			}
		}
		if(Imported.YEP_X_ItemUpgrades){
			if(Window_UpgradeItemList.prototype.containsType.call({"_item":this._targetItem}, this._item) && this._item.upgradeEffect.length > 0){
				var enabled = Window_UpgradeItemList.prototype.isEnabled.call({"_item":this._targetItem}, this._item);
				if(this._useCommands.includes('UPGRADE')) this.addCommand("Upgrade", 'upgrade', enabled);
			}
		}
		if(Imported.YEP_X_AttachAugments){
			if(this._targetItem.augmentSlots){
				ItemManager.checkAugmentSlots(this._targetItem);
				for (var i = 0; i < this._targetItem.augmentSlots.length; i++) {
					var enabled = this._targetItem.augmentSlotEnable[i];
					var fmt = Yanfly.Param.AugmentSlotFmt;
					var slot = this._targetItem.augmentSlots[i];
					if(Window_AugmentItemList.prototype.containsType.call({"_item":this._targetItem, "_slotId":i}, this._item)){
						var name = Window_ItemActionCommand.prototype.getAugmentSlotItemName.call({"_item":this._targetItem}, i);
						var text = fmt.format(slot, name);
						if(this._useCommands.includes('AUGMENT')) this.addCommand(text, 'augment', enabled, i);
					}
				}
				this.changePaintOpacity(true);
			}
		}
	}
	if(this._targetItem != null || this._item != null) this.updateHeight();
};

Window_ItemHeldCommand.prototype.setItem = function(item) {
	this._item = item;
};

Window_ItemHeldCommand.prototype.setTargetItem = function(item) {
	this._targetItem = item;
	this.refresh();
};

//-----------------------------------------------------------------------------
// Window_ConfirmCommand
//
// The window class of confirm commands.

function Window_ConfirmCommand() {
	this.initialize.apply(this, arguments);
}
Window_ConfirmCommand.prototype = Object.create(Window_Command.prototype);
Window_ConfirmCommand.prototype.constructor = Window_ConfirmCommand;
Window_ConfirmCommand.prototype.initialize = function() {
	var x = (Graphics.boxWidth - this.windowWidth()) / 2;
	var y = (Graphics.boxHeight - this.windowHeight()) / 2;
	Window_Command.prototype.initialize.call(this, x, y);
	this.textTitle = MRP.GridInventory.confirmDropWindowConfig.textTitle;
	this.textBody = MRP.GridInventory.confirmDropWindowConfig.textBody;
	this.refresh();
};

Window_ConfirmCommand.prototype.windowWidth = function() {
	//var textWidth = this.textBody ? this.textWidth(this.textBody) : 200;
	//return textWidth + this.standardPadding() * 2;
	return MRP.GridInventory.confirmDropWindowConfig.width;
};

Window_ConfirmCommand.prototype.drawAllItems = function() {
	Window_Command.prototype.drawAllItems.call(this);
	this.drawText(this.textTitle, 0, 0, this.contentsWidth(), "center");
	this.drawHorzLine(this.lineHeight())
	this.makeFontSmaller()
	var words = this.textBody ? this.textBody.split(" ") : [];
	var x = 0;
	var y = this.lineHeight();
	for(var i = 0; i < words.length; i++){
		var word = words[i];
		var wordHeight = this.calcTextHeight({"text":word, "index":0});
		var wordLength = this.textWidth(word + " ");
		if(x + wordLength > this.contentsWidth()) {
			x = 0;
			y += wordHeight;
		}
		this.drawText(word + " ", x, y, wordLength);
		x += wordLength;

	}
};

Window_ConfirmCommand.prototype.itemTextAlign = function() {
    return 'center';
};

Window_ConfirmCommand.prototype.drawHorzLine = function(y) {
	this.contents.paintOpacity = 48;
	this.contents.fillRect(0, y, this.contentsWidth(), 2, this.normalColor());
	this.contents.paintOpacity = 255;
};

Window_ConfirmCommand.prototype.itemRect = function(index) {
    var rect = new Rectangle();
    var maxCols = this.maxCols();
    rect.width = this.itemWidth();
    rect.height = this.itemHeight();
    rect.x = index % maxCols * (rect.width + this.spacing()) - this._scrollX;
	rect.y = Math.floor(index / maxCols) * rect.height - this._scrollY;
	
	//rect.y += (this.numVisibleRows() - 1) * this.lineHeight();

	rect.y = this.contentsHeight() - this.lineHeight();

    return rect;
};

Window_ConfirmCommand.prototype.numVisibleRows = function() {
    return 4;
};


Window_ConfirmCommand.prototype.maxCols = function() {
    return 2;
};

Window_ConfirmCommand.prototype.makeCommandList = function() {
	this.addCommand(MRP.GridInventory.confirmDropWindowConfig.textCancel, 'cancel')
	this.addCommand(MRP.GridInventory.confirmDropWindowConfig.textConfirm, 'confirm');
};

Window_ConfirmCommand.prototype.setTextTitle = function(text) {
	this.textTitle = text;
	this.refresh();
}

Window_ConfirmCommand.prototype.setTextBody = function(text) {
	this.textBody = text;
	this.refresh();
}

//-----------------------------------------------------------------------------
// Window_DisassemblePreview
//
// The window class for previewing what you would get from disassemble

function Window_DisassemblePreview() {
	this.initialize.apply(this, arguments);
}

Window_DisassemblePreview.prototype = Object.create(Window_ConfirmCommand.prototype);
Window_DisassemblePreview.prototype.constructor = Window_DisassemblePreview;

Window_DisassemblePreview.prototype.initialize = function() {
	Window_ConfirmCommand.prototype.initialize.call(this);
	this.textTitle = "Disassemble";
	this.textBody = '';
	this._types = ['ALL'];
	this._rates = [0];
	this._currentItem = null;
	this.refresh();
};

Window_DisassemblePreview.prototype.windowWidth = function() {
	return 500;
};

Window_DisassemblePreview.prototype.numVisibleRows = function() {
    return 10;
};

Window_DisassemblePreview.prototype.setItem = function(item){
	this._currentItem = new Game_GridItem(-1, item, 1);
	this.refresh();
}

Window_DisassemblePreview.prototype.drawAllItems = function() {
	Window_Command.prototype.drawAllItems.call(this);
	if(!this._currentItem) return;
	
	
	var itemTitle = Window_Tooltip.prototype.getTagFormat.call({'_gridItem':this._currentItem}, "formatTitle").format(this._currentItem.object().name);
	this.drawText(this.textTitle, 0, 0, this.contentsWidth() - this.textWidth(" " + itemTitle), "center");
	
	var rarity = MRP.GridInventory.rarities[this._currentItem.object().meta.Rarity];
	if(rarity) this.changeTextColor(rarity.Color);
	this.drawText(" " + itemTitle, this.textWidth(this.textTitle), 0, this.contentsWidth() - this.textWidth(this.textTitle), "center");
	
	this.resetFontSettings();


	this.drawHorzLine(this.lineHeight())
	
	if(this._currentItem){
		var dx = this.textPadding();
		var dy =this.lineHeight();
		var dw = this.contentsWidth() - dx * 2;
		this.changeTextColor(this.systemColor());
		var text = Yanfly.Param.IDADisList;
		this.drawText(text, dx, dy, dw, 'center');
		dy += this.lineHeight();
		dy = this.drawDisassembleItems(dx, dy);
		dy += this.standardPadding() * 2 + this.lineHeight() * 1.5;
		this.move(this.x, (Graphics.boxHeight - dy) / 2, this.width, dy)
	}
};

Window_DisassemblePreview.prototype.drawItemName = function(item, x, y, width) {
    width = width || 312;
    if (item) {
		var iconBoxWidth = item._width * (MRP.GridInventory.cellSize);
		var iconBoxHeight = item._height * (MRP.GridInventory.cellSize);

        this.resetTextColor();
        this.drawEquipmentIcon(item, x + 2, y + 2);
        this.drawText(item.object().name, x + iconBoxWidth + this.textPadding(), y + iconBoxHeight / 2 - this.standardFontSize() / 2, width - iconBoxWidth);
    }
};

Window_DisassemblePreview.prototype.drawEquipmentIcon = function(item, x, y){
	Window_ItemGrid.prototype.drawIcon.call(this, item, x, y, 1 * (MRP.GridInventory.cellSize), 1 * (MRP.GridInventory.cellSize));
}

Window_DisassemblePreview.prototype.drawIcon = function(item, x, y){
	item = new Game_GridItem(-1, $dataItems[item], 1);
	x -= this.textPadding();
	y -= this.textPadding();
	Window_ItemGrid.prototype.drawIcon.call(this, item, x, y, 1 * (MRP.GridInventory.cellSize), 1 * (MRP.GridInventory.cellSize));
}

Window_DisassemblePreview.prototype.setTypes = function(types, rates) {
    Window_DisassemblePool.prototype.setTypes.call(this, types, rates);
};

Window_DisassemblePreview.prototype.drawDisassembleItems = function(x, y) {
    var dx = x;
    var dy = y;
    var dw = this.contentsWidth() - dx * 2;
    var length = this._types.length;
    for (var i = 0; i < length; ++i) {
      var type = this._types[i];
      dy = this.drawDisassembleTypePool(dy, type);
	}
	if(dy == y) {
		var textWidth = this.textWidth('- None -')
		this.drawTextEx("\\c[7]- None -", (this.contentsWidth() - textWidth) / 2, dy)
		dy += this.lineHeight();
		this.resetFontSettings();
	}
	return dy;
};

Window_DisassemblePreview.prototype.drawDisassembleTypePool = function(dy, type) {
    var pool = ItemManager.getDisassembleItems(this._currentItem.object(), type);
    if (!pool) return dy;
    var index = this._types.indexOf(type);
    var rateBonus = this._rates[index] || 0;
    var length = pool.length;
    var fmt1 = Yanfly.Param.IDAQuantityFmt1;
    var fmt2 = Yanfly.Param.IDAQuantityFmt2;
    var dx = this.textPadding();
    this.resetFontSettings();
    var dw = this.contentsWidth() - dx * 2;
    for (var i = 0; i < length; ++i) {
      var data = pool[i];
      if (!data) continue;
      var item = data[0];
      var quantity1 = data[1][0];
      var quantity2 = data[1][1];
      if (quantity1 > quantity2) {
        quantity1 = quantity2;
        quantity2 = data[1][0];
      }
      quantity1 = Yanfly.Util.toGroup(quantity1);
      quantity2 = Yanfly.Util.toGroup(quantity2);
	  var rate = data[2] + rateBonus;
      var name = '\\i[' + item.id + ']' + item.name;
      if (quantity1 === quantity2) {
        var text = fmt1.format(quantity1, name);
      } else {
        var text = fmt2.format(quantity1, quantity2, name);
      }
      this.drawTextEx(text, dx, dy);
      rate = rate.clamp(0, 1);
      rate = (rate * 100).toFixed(0);
      text = rate + '%';
      this.contents.fontSize = Yanfly.Param.IDARateSize;
      this.drawText(text, dx, dy, dw, 'right');
      this.resetFontSettings();
      dy += this.lineHeight();
    }
    return dy;
};

//-----------------------------------------------------------------------------
// Window_Tooltip
//
// The sprite class of the held item.
function Window_Tooltip() {
	this.initialize.apply(this, arguments);
}
Window_Tooltip.prototype = Object.create(Window_Base.prototype);
Window_Tooltip.prototype.constructor = Window_Tooltip;
Window_Tooltip.prototype.initialize = function() {
	Window_Base.prototype.initialize.call(this, 0, 0, 0, 0);
	this.height = Graphics.boxHeight;
	//this._isWindow = false; // <- Should not be rendered as a window
	this._gridItem = null;
	this._item = null;
	this._widthNeeded = 0;
	this._shouldHide = true;
	this._compareWindow = null;
};

Window_Tooltip.prototype.setCompareWindow = function(window){
	this._compareWindow = window;
}

Window_Tooltip.prototype.show = function() {
	Window_Base.prototype.show.call(this);
	this._shouldHide = false;
	if(this._compareWindow) {
		if(HeldItemManager.isHoldingItem() && Input.isPressed("control")){
			this._compareWindow.setGridItem(HeldItemManager.heldItem());
			this._compareWindow.show();
		}
	}
};

Window_Tooltip.prototype.update = function() {
	Window_Base.prototype.update.call(this);
	if(this._shouldHide){
		this.setGridItem(null);
		this.hide();
	} else {
		this._shouldHide = true;
	}
};

Window_Tooltip.prototype.setPosition = function(x, y, rect) {
	if(this._compareWindow) this._compareWindow.setPosition(x < Graphics.boxWidth / 2 ? x + this.width : x - this._compareWindow.width, y + this._compareWindow.height < Graphics.boxHeight ? y : Graphics.boxHeight - this._compareWindow.height);
	if(this.x != x || this.y != y){
		this.move(x, y, this.width, this.height);
	}
};

Window_Tooltip.prototype.setGridItem = function(gridItem){
	if(gridItem == null || gridItem._index == -1 || gridItem.isEmpty()){
		this._gridItem = null;
		this._item = null;
		return;
	}
	if(this._gridItem != gridItem){
		this._gridItem = gridItem;
		this._item = gridItem.object();
		this._widthNeeded = 0;
		this.refresh(); // To get right width
		this.refresh(); // TO draw with the right width
	}
}
Window_Tooltip.prototype.standardFontSize = function() {
	return 14;
};
Window_Tooltip.prototype.refresh = function() {
	this.createContents();
	if(this._gridItem && !this._gridItem.isEmpty()){
		var dataClass = this._gridItem.getDataClass(this._gridItem.object());
		if(dataClass == 'item'){
			this.drawItem();
		} else {
			this.drawEquipment();
		}
	}
};

Window_Tooltip.prototype.isTagVisibile = function(tag){
	var item = this._gridItem.object();
	if(item.toolTipVisibility[tag] != undefined){
		return item.toolTipVisibility[tag];
	} else {
		return eval("MRP.GridInventory.toolTipConfig." + tag);
	}
}

Window_Tooltip.prototype.getTagFormat = function(tag){
	var item = this._gridItem.object();
	if(item.toolTipFormat[tag] != undefined){
		return item.toolTipFormat[tag];
	} else {
		return eval("MRP.GridInventory.toolTipConfig." + tag)
	}
}

Window_Tooltip.prototype.drawItem = function(){
	var width = this.contents.width;
	var y = 0;
	var lastHorzY = 0;
	if(this.isTagVisibile("showTitle")) y = this.drawTitle(0, y, width);
	if(this.isTagVisibile("showType")) y = this.drawType(0, y, width);
	if(this.isTagVisibile("showMaxItems")) y = this.drawMaxItems(0, y, width);
	if(lastHorzY != y) {
		y = this.drawHorzLine(y + 5);
		lastHorzY = y;
	}
	if(this.isTagVisibile("showDamage")) y = this.drawDamage(0, y, width);
	if(lastHorzY != y) {
		y = this.drawHorzLine(y + 5);
		lastHorzY = y;
	}
	if(this.isTagVisibile("showItemEffects")) y = this.drawEffects(0, y, width);
	if(lastHorzY != y) {
		y = this.drawHorzLine(y + 5);
		lastHorzY = y;
	}
	if(Imported.YEP_X_ItemUpgrades){
		y = this.drawItemUpgrades(0, y, width)
		if(lastHorzY != y) {
			y = this.drawHorzLine(y + 5);
			lastHorzY = y;
		}
	}
	y = this.drawDescription(0, y, width);
	this.setHeight(y);
}
Window_Tooltip.prototype.drawDamage = function(x, y, width) {
	var damage = this._gridItem.object().damage;
	if(damage.type == 0) return y;
	var amount = Number(damage.formula);
	if(amount) {
		var element = $dataSystem.elements[damage.elementId];
		var variance = damage.variance;
		var minDmg = amount;
		var maxDmg = amount;
		if(variance != 0) {
			var varianceAmount = amount * (variance/100);
			minDmg -= varianceAmount;
			maxDmg += varianceAmount;
			amount = Math.floor(minDmg) + "-" + Math.floor(maxDmg);
		}
		
		var text = this.getTagFormat("formatDamage[" + String(damage.type-1) + "]").format(amount, element);
		this.drawText(text, x, y, width, 'center');
		return y + this.lineHeight();
	} else {
		return y;
	}
}

Window_Tooltip.prototype.effectFormatter = function(text, percentValue, flatValue){
	var percent = "";
	var flat = "";
	var percentPos = "";
	var flatPos = "";
	var both = "";
	
	if(percentValue != 0) {
		percentPos = percentValue < 0 ? MRP.GridInventory.toolTipConfig.effectHelper.textLose : MRP.GridInventory.toolTipConfig.effectHelper.textRecover;
		percent = Math.abs(percentValue) * 100 + "%";
		if(flatValue != 0) both = MRP.GridInventory.toolTipConfig.effectHelper.textBoth;
	}
	if(flatValue != 0)  {
		flatPos = flatValue < 0 ? MRP.GridInventory.toolTipConfig.effectHelper.textLose : MRP.GridInventory.toolTipConfig.effectHelper.textRecover;
		flat = Math.abs(flatValue)
	}

	return text.format(percent, flat, percentPos, flatPos, both);
}

Window_Tooltip.prototype.drawEffects = function(x, y, width) {
	this.changeTextColor(this.systemColor());
	var effects = this._gridItem.object().effects;
	for(var i = 0; i < effects.length; i++){
		var string = "";
		var effect = effects[i];
		switch(effect.code) {
			case 11:
				if(!this.isTagVisibile("showEffect11")) break;
				string = this.effectFormatter(this.getTagFormat("formatEffect11"), effect.value1, effect.value2);
				break;
			case 12:
				if(!this.isTagVisibile("showEffect12")) break;
				string = this.effectFormatter(this.getTagFormat("formatEffect12"), effect.value1, effect.value2);
				break;
			case 13:
				if(!this.isTagVisibile("showEffect13")) break;
				string = this.effectFormatter(this.getTagFormat("formatEffect13"), 0, effect.value1);
				break;
			case 21:
				if(!this.isTagVisibile("showEffect21")) break;
				string = this.getTagFormat("formatEffect21").format($dataStates[effect.dataId].name, (effect.value1 * 100) + "%");
				break;
			case 22:
				if(!this.isTagVisibile("showEffect22")) break;
				string = this.getTagFormat("formatEffect22").format($dataStates[effect.dataId].name, (effect.value1 * 100) + "%");
				break;
			case 31:
				if(!this.isTagVisibile("showEffect31")) break;
				string = this.getTagFormat("formatEffect31").format(TextManager.param(effect.dataId), effect.value1);
				break;
			case 32:
				if(!this.isTagVisibile("showEffect32")) break;
				string = this.getTagFormat("formatEffect32").format(TextManager.param(effect.dataId), effect.value1);
				break;
			case 33:
				if(!this.isTagVisibile("showEffect33")) break;
				string = this.getTagFormat("formatEffect33").format(TextManager.param(effect.dataId), effect.value1);
				break;
			case 34:
				if(!this.isTagVisibile("showEffect34")) break;
				string = this.getTagFormat("formatEffect34").format(TextManager.param(effect.dataId), effect.value1);
				break;
			case 41:
				if(!this.isTagVisibile("showEffect41")) break;
				string = this.getTagFormat("formatEffect41");
				break;
			case 42:
				if(!this.isTagVisibile("showEffect42")) break;
				string = this.getTagFormat("formatEffect42").format(TextManager.param(effect.dataId), effect.value1);
				break;
			case 43:
				if(!this.isTagVisibile("showEffect43")) break;
				string = this.getTagFormat("formatEffect43").format($dataSkills[effect.dataId].name);
				break;
		}
		if(string != ""){
			this.drawText(string, x, y, width, 'center');
			y += this.lineHeight();
		}
	}
	this.resetFontSettings();
	return y;
}
Window_Tooltip.prototype.drawMaxItems = function(x, y, width) {
	var current = this._gridItem._amount;
	var max = this._gridItem.maxItems();
	if(max > 1) {
		var text = this.getTagFormat("formatMaxItems").format(current, max);
		this.drawText(text, x, y, width, 'center');
		return y + this.lineHeight();	
	} else {
		return y;
	}
}

if(Imported.YEP_X_ItemDurability){
	Window_Tooltip.prototype.drawDurability = function(x, y, width) {
		var item = this._gridItem.object();
		if(!Window_ItemInfo.prototype.isDrawDurability.call({"_item":item})) return y;
		var cur = DataManager.getDurability(item);
		var max = DataManager.getMaxDurability(item);
		
		var text = "";
		if(cur > 0){
			var fmt = Yanfly.Param.IDurFmt;
			var color = Window_ItemInfo.prototype.durabilityColor.call({"_item":item}, cur, max);
			text = Yanfly.Param.IDurText + " " + fmt.format(cur, max);
			this.changeTextColor(this.textColor(color));
		} else {
			text = Yanfly.Param.IDurUnbreakable;
			this.changeTextColor(this.textColor(Yanfly.Param.IDurColor['unbreak']));
		}
		this.drawText(text, x, y, width, "center");
		
		this.resetFontSettings();
		return y + this.lineHeight();
	}
}

Window_Tooltip.prototype.drawEquipment = function(){
	var width = this.contents.width;
	var y = 0;
	var lastHorzY = y;
	if(this.isTagVisibile("showTitle")) y = this.drawTitle(0, y, width);
	if(this.isTagVisibile("showType")) y = this.drawType(0, y, width);
	if(this.isTagVisibile("showMaxItems")) y = this.drawMaxItems(0, y, width);
	if(Imported.YEP_X_ItemDurability && this.isTagVisibile("showDurability")) y = this.drawDurability(0, y, width);
	if(lastHorzY != y) {
		y = this.drawHorzLine(y + 5);
		lastHorzY = y;
	}
	if(this.isTagVisibile("showParameters")) y = this.drawParameters(0, y, width);
	if(lastHorzY != y) {
		y = this.drawHorzLine(y + 5);
		lastHorzY = y;
	}
	y = this.drawTraits(0, y, width);
	if(lastHorzY != y) {
		y = this.drawHorzLine(y + 5);
		lastHorzY = y;
	}
	if(Imported.YEP_X_ItemUpgrades && this.isTagVisibile("showUpgrades")){
		y = this.drawItemUpgrades(0, y, width)
		if(lastHorzY != y) {
			y = this.drawHorzLine(y + 5);
			lastHorzY = y;
		}
	}
	if(Imported.YEP_X_AttachAugments && this.isTagVisibile("showAugments")){
		y = this.drawAugments(0, y, width)
		if(lastHorzY != y) {
			y = this.drawHorzLine(y + 5);
			lastHorzY = y;
		}
	}
	if(this.isTagVisibile("showDescription")) y = this.drawDescription(0, y, width);
	this.setHeight(y);
}

if(Imported.YEP_X_ItemUpgrades){
	Window_Tooltip.prototype.drawItemUpgrades = function(x, y, width) {
		var item = this._gridItem.object();
		var baseItem = DataManager.getBaseItem(item);
		// Slots Available
		if (Yanfly.Param.IUSSlotsText === '') return y;
		var textMaxSlots;
		if (item.originalUpgradeSlots) {
			if(item.originalUpgradeSlots == 0) return y;
			textMaxSlots = '/' + Yanfly.Util.toGroup(item.originalUpgradeSlots);
		} else {
			if(item.upgradeSlots == 0) return y;
			textMaxSlots = '/' + Yanfly.Util.toGroup(baseItem.upgradeSlots);
		}
		var text = Yanfly.Param.IUSSlotsText + " " + Yanfly.Util.toGroup(item.upgradeSlots) + textMaxSlots;
		if (item.upgradeSlots <= 0) this.changeTextColor(this.powerDownColor());
		this.drawText(text, x, y, width, "center");
		y += this.lineHeight()

		// Applied Upgrades
		var fmt = Yanfly.Param.IUSSlotFmt;
		for (var i = 0; i < item.slotsApplied.length; ++i) {
			var text = fmt.format(i + 1, item.slotsApplied[i]);
			this.drawTextEx(text, x, y); // Actually draw it
			y += this.lineHeight();
		}
		this.resetFontSettings();
		return y;
	}
}

if(Imported.YEP_X_AttachAugments){
	Window_Tooltip.prototype.drawAugments = function(x, y, width) {
		y = Window_ItemInfo.prototype.drawAugmentInfo.call(this, y);
		this.resetFontSettings();
		return y;
	}

	Window_Tooltip.prototype.drawAugmentData = function(slot, dy){
		Window_ItemInfo.prototype.drawAugmentData.call(this, slot, dy);
	}
}


Window_Tooltip.prototype.processDrawIcon = function(iconIndex, textState) {
    this.drawIcon(iconIndex, textState.x + 2, textState.y + 2);
    textState.x += this.standardFontSize() + 4;
};

Window_Tooltip.prototype.drawIcon = function(iconIndex, x, y) {
    var bitmap = ImageManager.loadSystem('IconSet');
    var pw = Window_Base._iconWidth;
    var ph = Window_Base._iconHeight;
    var sx = iconIndex % 16 * pw;
    var sy = Math.floor(iconIndex / 16) * ph;
    this.contents.blt(bitmap, sx, sy, pw, ph, x, y, this.standardFontSize(), this.standardFontSize());
};


Window_Tooltip.prototype.drawTraits = function(x, y, width) {
	var item = this._gridItem.object();
	if(!item.params) return y;
	this.changeTextColor(this.systemColor());
	for(var i = 0; i < item.traits.length; i++){
		var trait = item.traits[i];
		var string = "";
		switch(trait.code){
			case 11:
				if(!this.isTagVisibile("showTrait11")) break;
				var element = $dataSystem.elements[trait.dataId];
				var value = trait.value < 1 ? "-" + Math.ceil((1 - trait.value) * 100) : "+" + Math.ceil((trait.value - 1) * 100) + "%";
				string = this.getTagFormat("formatTrait11").format(element, value);
				break;
			case 12:
				if(!this.isTagVisibile("showTrait12")) break;
				var debuff = TextManager.param(trait.dataId);
				var value = trait.value < 1 ? "-" + Math.ceil((1 - trait.value) * 100) : "+" + Math.ceil((trait.value - 1) * 100) + "%";
				string = this.getTagFormat("formatTrait12").format(debuff, value);
				break;
			case 13:
				if(!this.isTagVisibile("showTrait13")) break;
				var state = $dataStates[trait.dataId].name;
				var value = trait.value < 1 ? "-" + Math.ceil((1 - trait.value) * 100) : "+" + Math.ceil((trait.value - 1) * 100) + "%";
				string = this.getTagFormat("formatTrait13").format(state, value);
				break;
			case 14:
				if(!this.isTagVisibile("showTrait14")) break;
				var state = $dataStates[trait.dataId].name;
				string = this.getTagFormat("formatTrait14").format(state);
				break;
			case 21:
				if(!this.isTagVisibile("showTrait21")) break;
				var param = TextManager.param(trait.dataId);
				var value = trait.value < 1 ? "-" + Math.ceil((1 - trait.value) * 100) : "+" + Math.ceil((trait.value - 1) * 100) + "%";
				string = this.getTagFormat("formatTrait21").format(param, value);
				break;
			case 22:
				if(!this.isTagVisibile("showTrait22")) break;
				var exParams = ["Hit Rate", "Evasion Rate", "Critical Rate", "Critical Evasion", "Magic Evasion", "Magic Reflection", "Counter Attack", "HP Regeneration", "MP Regeneration", "TP Regeneration"];
				var exParam = exParams[trait.dataId];
				string = this.getTagFormat("formatTrait22").format(exParam, trait.value * 100 + "%");
				break;
			case 23:
				if(!this.isTagVisibile("showTrait23")) break;
				var spParams = ["Target Rate", "Guard Rate", "Recovery Effect", "Pharmacology", "MP Cost Rate", "TP Charge Rate", "Counter Attack", "Physical Damage", "Magical Damage", "Expeience"];
				var spParam = spParams[trait.dataId];
				var value = trait.value < 1 ? "-" + Math.ceil((1 - trait.value) * 100) : "+" + Math.ceil((trait.value - 1) * 100) + "%";
				string = this.getTagFormat("formatTrait23").format(spParam, value);
				break;
			case 31:
				if(!this.isTagVisibile("showTrait24")) break;
				var element = $dataSystem.elements[trait.dataId];
				string = this.getTagFormat("formatTrait24").format(element);			
				break;
			case 32:
				if(!this.isTagVisibile("showTrait32")) break;
				var state = $dataStates[trait.dataId].name;
				var value = trait.value < 1 ? "-" + Math.ceil((1 - trait.value) * 100) : "+" + Math.ceil((trait.value - 1) * 100) + "%";
				string = this.getTagFormat("formatTrait32").format(state, value);
				break;
			case 33:
				if(!this.isTagVisibile("showTrait33")) break;
				string = this.getTagFormat("formatTrait33").format(trait.value * 100);
				break;
			case 34:
				if(!this.isTagVisibile("showTrait34")) break;
				string = this.getTagFormat("formatTrait34").format(trait.value);
				break;
			case 41:
				if(!this.isTagVisibile("showTrait41")) break;
				string = this.getTagFormat("formatTrait41").format($dataSystem.skillTypes[trait.dataId]);
				break;
			case 42:
				if(!this.isTagVisibile("showTrait42")) break;
				string = this.getTagFormat("formatTrait42").format($dataSystem.skillTypes[trait.dataId]);
				break;
			case 43:
				if(!this.isTagVisibile("showTrait43")) break;
				string = this.getTagFormat("formatTrait43").format($dataSkills[trait.dataId].name);
				break;
			case 44:
				if(!this.isTagVisibile("showTrait44")) break;
				string = this.getTagFormat("formatTrait44").format($dataSkills[trait.dataId].name);
				break;
			case 51:
				if(!this.isTagVisibile("showTrait51")) break;
				string = this.getTagFormat("formatTrait51").format($dataWeapons[trait.dataId].name);
				break;
			case 52:
				if(!this.isTagVisibile("showTrait52")) break;
				string = this.getTagFormat("formatTrait51").format($dataArmors[trait.dataId].name);
				break;
			case 53:
				if(!this.isTagVisibile("showTrait53")) break;
				string = this.getTagFormat("formatTrait53").format($dataSystem.equipTypes[trait.dataId]);
				break;
			case 54:
				if(!this.isTagVisibile("showTrait54")) break;
				string = this.getTagFormat("formatTrait54").format($dataSystem.equipTypes[trait.dataId]);
				break;
			case 55:
				if(!this.isTagVisibile("showTrait55")) break;
				string = this.getTagFormat("formatTrait55").format((trait.dataId == 0 ? "Normal" : "Dual Wield"));
				break;
			case 61:
				if(!this.isTagVisibile("showTrait61")) break;
				this.getTagFormat("formatTrait61").format((trait.value * 100) + "%");
				break;
			case 62:
				if(!this.isTagVisibile("showTrait62")) break;
				this.getTagFormat("formatTrait62[" + String(trait.dataId) + "]");
				break;
		}
		if(string != ""){
			this.drawText(string, x, y, width, 'center');
			y += this.lineHeight(); 
		}
	}
	this.resetFontSettings();
	return y;
}
Window_Tooltip.prototype.drawText = function(text, x, y, maxWidth, align) {
	var widthNeeded = this.textWidth(text);
	if(this._widthNeeded < widthNeeded) {
		this._widthNeeded = widthNeeded;
		this.width = widthNeeded + this.standardPadding() * 3;
		this.contents.width = this.contentsWidth();
	}
	Window_Base.prototype.drawText.call(this, text, x, y, maxWidth, align)
};

Window_Tooltip.prototype.drawTextEx = function(text, x, y) {
	x = 0;
	var width = this.contents.width;
	var widthNeeded = Window_Base.prototype.drawTextEx.call(this, text, x, -10000); // Find width
	Window_Base.prototype.drawTextEx.call(this, text, x + (width - widthNeeded) / 2, y) // Actually draw
	if(this._widthNeeded < widthNeeded) {
		this._widthNeeded = widthNeeded;
		this.width = widthNeeded + this.standardPadding() * 3;
		this.contents.width = this.contentsWidth();
	}
	return widthNeeded;
};


Window_Tooltip.prototype.showParameters = function(i){
	switch(i) {
		case 0:
			return this.isTagVisibile("showMaxHP");
		case 1:
			return this.isTagVisibile("showMaxMP");
		case 2:
			return this.isTagVisibile("showAttack");
		case 3:
			return this.isTagVisibile("showDefense");
		case 4:
			return this.isTagVisibile("showMAttack");
		case 5:
			return this.isTagVisibile("showMDefense");
		case 6:
			return this.isTagVisibile("showAgility");
		case 7:
			return this.isTagVisibile("showLuck");
	}
	return false;
}

Window_Tooltip.prototype.drawParameters = function(x, y, width) {
	var item = this._gridItem.object();
	if(!item.params) return y;
	this.changeTextColor(this.systemColor());
	for(var i = 0; i < item.params.length; i++){
		if(!this.showParameters(i)) continue;
		var value = item.params[i];
		if(value != 0){
			string = this.getTagFormat("formatParameters").format(TextManager.param(i), (value > 0 ? "+" : "") + value)
			this.drawText(string, x, y, width, 'center');
			y += this.lineHeight();	
		}
	}
	this.resetFontSettings();
	return y;
}
Window_Tooltip.prototype.drawHorzLine = function(y) {
	this.contents.paintOpacity = 48;
	this.contents.fillRect(0, y, this.contentsWidth(), 2, this.normalColor());
	this.contents.paintOpacity = 255;
	return y + 5;
};
Window_Tooltip.prototype.lineHeight = function() {
	return 18;
};
Window_Tooltip.prototype.setHeight = function(contentHeight) {
	this.height = contentHeight + this.standardPadding() * 2;
};
Window_Tooltip.prototype.drawTitle = function(x, y, width) {
	this.contents.fontSize = 22;
	var rarity = MRP.GridInventory.rarities[this._gridItem.object().meta.Rarity];
	if(rarity) this.changeTextColor(rarity.Color);

	var text = this.getTagFormat("formatTitle").format(this._gridItem.object().name);
	this.drawText(text, x, y, width, 'center');	
	this.resetFontSettings();
	return y + this.lineHeight() * 1.5;
}
Window_Tooltip.prototype.drawType = function(x, y, width) {
	var dataClass = this._gridItem.getDataClass(this._gridItem.object());
	var type = "";
	switch(dataClass) {
		case 'weapon':
			type = $dataSystem.weaponTypes[this._gridItem.object().wtypeId];
			break;
		case 'armor':
			type = $dataSystem.armorTypes[this._gridItem.object().atypeId];
			break;
		case 'item':
			type = "Item";
	}
	var text = this.getTagFormat("formatType").format(type);
	this.drawText(text, x, y, width, 'center');
	return y + this.lineHeight();		
}
Window_Tooltip.prototype.drawDescription = function(x, y, width) {
	this.drawText(this._gridItem.object().description, x, y, width, 'center');	
	return y + this.lineHeight();
}

//-----------------------------------------------------------------------------
// Sprite_HeldItem
//
// The sprite class of the held item.
function Sprite_HeldItem() {
	this.initialize.apply(this, arguments);
}
Sprite_HeldItem.prototype = Object.create(Window_Base.prototype);
Sprite_HeldItem.prototype.constructor = Sprite_HeldItem;
Sprite_HeldItem.prototype.initialize = function(gridWidth, gridHeight) {
	Window_Base.prototype.initialize.call(this, 0, 0, gridWidth, gridHeight);
	this._gridItem = null;
	this._gridWidth = gridWidth;
	this._gridHeight = gridHeight;
	this._isWindow = false;
	this.opacity = 0;
};

Sprite_HeldItem.prototype.contentsWidth = function() {
    return this.width;
};

Sprite_HeldItem.prototype.contentsHeight = function() {
    return this.height;
};

/*
Sprite_HeldItem.prototype.hasGridItem = function() {
	return this._gridItem != null;
}
*/

Sprite_HeldItem.prototype.setGridItem = function(gridItem){
	if(gridItem == null || gridItem._index == -1 || gridItem.isEmpty()){
		this._gridItem = null;
		return;
	}
	if(this._gridItem != gridItem) {
		this._gridItem = gridItem;
		var width = this._gridWidth * gridItem._width;
		var height = this._gridHeight * gridItem._height;
		this.move(this.x, this.y, width, height);

		if(TouchInput.isPressed()) {
			var x = TouchInput._mouseX - (this._gridWidth/2 * this._gridItem._width);
			var y = TouchInput._mouseY - (this._gridHeight/2 * this._gridItem._height);
			HeldItemManager.setPosition(x, y);
		}

		this.updateBitmap();
	}
}

Sprite_HeldItem.prototype.update = function() {
	Window_Base.prototype.update.call(this);
	this.setGridItem(HeldItemManager.heldItem());
	if (this.contents) {
		this.contents.clear();	
		this.updatePosition();
        if (this._gridItem != null){
			this.drawIcon(this._gridItem, 0, 0, this._gridItem._width * Window_Base._iconWidth, this._gridItem._height * Window_Base._iconHeight);
		} 
    }
};

Sprite_HeldItem.prototype.updatePosition = function() {
	if (this._gridItem != null){
		if(TouchInput._mouseMoved) {
			var x = TouchInput._mouseX - (this._gridWidth/2 * this._gridItem._width);
			var y = TouchInput._mouseY - (this._gridHeight/2 * this._gridItem._height);
			HeldItemManager.setPosition(x, y);
		}
	}
	var pos = HeldItemManager.getPosition();
	this.move(pos.x, pos.y, this.width, this.height);
}

Sprite_HeldItem.prototype.updateBitmap = function() {
	this.contents = new Bitmap(this.width, this.height);
}

Sprite_HeldItem.prototype.drawIcon = function(item, x, y, width, height) {
	Window_ItemGrid.prototype.drawIcon.call(this, item, x, y, width, height);
	this.makeFontSmaller();
	Window_ItemGrid.prototype.drawItemNumber.call(this, item, x + this.textPadding(), y - this.textPadding() * 2, width);
	this.resetFontSettings();
}

Sprite_HeldItem.prototype.textPadding = function() {
    return 4;
};

Sprite_HeldItem.prototype.standardPadding = function() {
    return 0;
};

Sprite_HeldItem.prototype.needsNumber = function(item) {
	return Window_ItemGrid.prototype.needsNumber.call(this, item);
};

//-----------------------------------------------------------------------------
// Scene_InventoryBase
//
// The scene class of the grid item screen.
function Scene_InventoryBase() {
	this.initialize.apply(this, arguments);
}

MRP.GridInventory.Scene_InventoryBase = Scene_InventoryBase;
Scene_InventoryBase.prototype = Object.create(Scene_ItemBase.prototype);
Scene_InventoryBase.prototype.constructor = Scene_InventoryBase;
Scene_InventoryBase.prototype.initialize = function() {
	Scene_ItemBase.prototype.initialize.call(this);
	this._activeItem = null;
	this._activeItemWindow = null;
};
Scene_InventoryBase.prototype.create = function() {
	Scene_ItemBase.prototype.create.call(this);
	//this.createHelpWindow();
	this.createInventoryWindows();	
	this.createToolTip();
	if(Imported.YEP_X_ItemDisassemble) this.createDisassembleWindows();
	this.createCommandWindow();
	this.createActionCommandWindow();
	this.createDropCommandWindow();
	this.createActorWindow();
};

Scene_InventoryBase.prototype.isWithinWindows = function(){
	if(this._itemWindow.isOpen() && this._itemWindow.isTouchedInsideFrame()){
		return true;
	}
	if(this._commandWindow.isOpen() && this._commandWindow.isTouchedInsideFrame()){
		return true;
	}
	if(this._heldCommandWindow.isOpen() && this._heldCommandWindow.isTouchedInsideFrame()){
		return true;
	}
	if(this._dropCommandWindow.isOpen() && this._dropCommandWindow.isTouchedInsideFrame()){
		return true;
	}
	return false;
}

Scene_InventoryBase.prototype.createInventoryWindows = function() {
	this.createHeldItemSprite();
	this.createPlayerInventoryWindow();
};
Scene_InventoryBase.prototype.createPlayerInventoryWindow = function() {
	var wh = 280;
	var ww = Graphics.boxWidth / 2 + 100;
	var wy = Graphics.boxHeight - wh;
	var wx = (Graphics.boxWidth - ww) / 2;
	this._itemWindow = new Window_ItemGrid(wx, wy, ww, wh);
	//this._itemWindow.setHelpWindow(this._helpWindow);
	this._itemWindow.setHandler('ok',     this.onItemOk.bind(this, this._itemWindow));
	this._itemWindow.setHandler('cancel', this.popScene.bind(this));
	this.addWindow(this._itemWindow);
	this._activeItemWindow = this._itemWindow;
	this._itemWindow.activateKeyboard();
	this._itemWindow.setPlayerInventory(true);
	this._itemWindow.activate();
};
Scene_InventoryBase.prototype.createHeldItemSprite = function() {
	var ww = MRP.GridInventory.cellSize; // GridItem width
	var wh = MRP.GridInventory.cellSize; // GridItem height
	this._heldItemWindow = new Sprite_HeldItem(ww, wh);
	this.addWindow(this._heldItemWindow);
}
Scene_InventoryBase.prototype.createToolTip = function() {
	this._toolTipWindow = new Window_Tooltip();
	this._compareWindow = new Window_Tooltip();
	this._toolTipWindow.setCompareWindow(this._compareWindow);
	this._itemWindow.setToolTipWindow(this._toolTipWindow);
	this.addWindow(this._toolTipWindow);
	this.addWindow(this._compareWindow);
}
Scene_InventoryBase.prototype.createCommandWindow = function() {
	this._commandWindow = new Window_ItemCommand(0, 0);
	this._commandWindow.setHandler('use',      this.commandUse.bind(this));
	this._commandWindow.setHandler('move',     this.commandMove.bind(this));
	this._commandWindow.setHandler('drop',     this.commandDrop.bind(this));
	if(Imported.YEP_X_ItemRename) this._commandWindow.setHandler('rename', this.commandRename.bind(this));
	if(Imported.YEP_X_AttachAugments) this._commandWindow.setHandler('augment', this.commandAugment.bind(this));
	this._commandWindow.setHandler('cancel',   this.commandCancel.bind(this));
	this.addWindow(this._commandWindow);
	this._commandWindow.close();
	this._commandWindow.openness = 0;
}

Scene_InventoryBase.prototype.createActionCommandWindow = function() {
	this._heldCommandWindow = new Window_ItemHeldCommand(0, 0);
	this._heldCommandWindow.setHandler('swap',      this.commandSwap.bind(this));
	if(Imported.YEP_X_ItemDisassemble) this._heldCommandWindow.setHandler('disassemble', this.commandDisassemble.bind(this));
	if(Imported.YEP_X_ItemDurability) this._heldCommandWindow.setHandler('repair', this.commandRepair.bind(this));
	if(Imported.YEP_X_ItemUpgrades) this._heldCommandWindow.setHandler('upgrade', this.commandUpgrade.bind(this));
	if(Imported.YEP_X_AttachAugments) this._heldCommandWindow.setHandler('augment', this.heldCommandAugment.bind(this));
	this._heldCommandWindow.setHandler('cancel',      this.commandCancel.bind(this));
	this.addWindow(this._heldCommandWindow);
	this._heldCommandWindow.close();
	this._heldCommandWindow.openness = 0;
}

if(Imported.YEP_X_ItemRename){
	Scene_InventoryBase.prototype.commandRename = function() {
		var item = this.item();
		SceneManager.push(Scene_ItemRename);
		SceneManager.prepareNextScene(item);
	};
}

if(Imported.YEP_X_ItemDisassemble){

	Scene_InventoryBase.prototype.createDisassembleWindows = function(){
		this._disassemblePoolWindow = new Window_DisassemblePreview();
    	this.addWindow(this._disassemblePoolWindow);
		this._disassemblePoolWindow.close();
		this._disassemblePoolWindow.setHandler('confirm', this.commandDisassembleConfirm.bind(this));
		this._disassemblePoolWindow.setHandler('cancel', this.commandDisassembleCancel.bind(this));
		this._disassemblePoolWindow.close();
		this._disassemblePoolWindow.openness = 0;
	}

	Scene_InventoryBase.prototype.commandDisassembleConfirm = function(){
		var itemWindow = this._activeItemWindow;
		var effectItem = this._heldCommandWindow._item;
		var targetItem = this._heldCommandWindow._targetItem;

		$gameParty.loseItem(targetItem, 1);

		if (effectItem) {
			var results = ItemManager.disassemble(targetItem, effectItem);
		  } else {
			var results = [];
		  }
		  itemWindow.refresh();
		  HeldItemManager.loseHeldItem(1);
		  this._disassemblePoolWindow.close();
		  //this._disassembleResultWindow.setResults(results)
		  this.activateItemWindow();
	}

	Scene_InventoryBase.prototype.commandDisassembleCancel = function(){
		this._disassemblePoolWindow.close();
		this.activateItemWindow();
	}

	Scene_InventoryBase.prototype.commandDisassemble = function() {
		var itemWindow = this._activeItemWindow;
		var effectItem = this._heldCommandWindow._item;
		var targetItem = this._heldCommandWindow._targetItem;

		this.deactivateItemWindow();
		this._heldCommandWindow.close();
		this._disassemblePoolWindow.setItem(targetItem);
		this._disassemblePoolWindow.setTypes(effectItem.disassemblerTypes, effectItem.disassemblerRates);
		this._disassemblePoolWindow.open();
		this._disassemblePoolWindow.activate();

		/*
		if (effectItem) {
		  var results = ItemManager.disassemble(targetItem, effectItem);
		} else {
		  var results = [];
		}
		$gameParty.loseItem(targetItem, 1);
		itemWindow.refresh();
		HeldItemManager.loseHeldItem(1);
		this._heldCommandWindow.close();
		//this._disassembleResultWindow.setResults(results)
		this.activateItemWindow();
		*/
	};
}

if(Imported.YEP_X_ItemDurability){
	Scene_InventoryBase.prototype.commandRepair = function() {
		var itemWindow = this._activeItemWindow;
		var effectItem = this._heldCommandWindow._item;
		var targetItem = this._heldCommandWindow._targetItem;

		ItemManager.applyRepairDurabilityEffects(targetItem, effectItem);
		Scene_Item.prototype.onRepairEval.call({"._item":targetItem}, effectItem);
		if (effectItem.consumable) HeldItemManager.loseHeldItem(1);
		this._heldCommandWindow.close();
		this.activateItemWindow();
	};
}

if(Imported.YEP_X_ItemUpgrades){
	Scene_InventoryBase.prototype.commandUpgrade = function() {
		var effectItem = this._heldCommandWindow._item;
		var targetItem = this._heldCommandWindow._targetItem;

		ItemManager.applyIUSEffects(targetItem, effectItem)
		HeldItemManager.loseHeldItem(1);

		this._heldCommandWindow.close();
		this.activateItemWindow();
	};

	ItemManager.payIUSEffects = function(mainItem, effectItem) {
		mainItem.upgradeSlots -= effectItem.upgradeSlotCost;
		this.addIUSLine(mainItem, effectItem);
		for (var i = 1; i < effectItem.upgradeSlotCost; ++i) {
		  mainItem.slotsApplied.push('---');
		}
		this.increaseItemBoostCount(mainItem, effectItem.boostCountValue);
	};
}

if(Imported.YEP_X_AttachAugments){

	// Detach Augment
	Scene_InventoryBase.prototype.commandAugment = function() {
		var item = this._commandWindow._item;
		var slotId = this._commandWindow.currentExt();

		var augment = ItemManager.removeAugmentFromSlot(item, slotId);
		ItemManager.applyAugmentEffects(item, null, slotId);
		HeldItemManager.setHeldItem(new Game_GridItem(1, augment, 1));
		this._commandWindow.close();
		this.activateItemWindow();
	}

	// Attach Augment
	Scene_InventoryBase.prototype.heldCommandAugment = function() {
		var effectItem = this._heldCommandWindow._item;
		var targetItem = this._heldCommandWindow._targetItem;
		var slotId = this._heldCommandWindow.currentExt();

		ItemManager.applyAugmentEffects(targetItem, effectItem, slotId, 1);
		HeldItemManager.loseHeldItem(1);

		this._heldCommandWindow.close();
		this.activateItemWindow();
	};

	ItemManager.applyAugmentEffects = function(item, effectItem, slotId, gain) {
		if (!item) return;
		gain = gain || 0;
		this.checkAugmentSlots(item);
		if (item.augmentSlotItems[slotId] !== 'none') {
		  var augment = this.removeAugmentFromSlot(item, slotId);
		  if (augment) $gameParty.gainItem(augment, gain);
		}
		this.installAugmentToSlot(item, effectItem, slotId);
		this.augmentRefreshParty(item);
	};

}

Scene_InventoryBase.prototype.createDropCommandWindow = function() {
	var ww = 100;
	var wh = 100;
	var x = Graphics.boxWidth / 2 - ww;
	var y = Graphics.boxHeight / 2 - wh;
	this._dropCommandWindow = new Window_ConfirmCommand(x, y, ww, wh);
	this._dropCommandWindow.setHandler('confirm', this.commandConfirmDrop.bind(this));
	this._dropCommandWindow.setHandler('cancel',  this.commandCancel.bind(this));
	this.addWindow(this._dropCommandWindow);
	this._dropCommandWindow.close();
	this._dropCommandWindow.openness = 0;
}

Scene_InventoryBase.prototype.commandConfirmDrop = function(){
	// Depending on option it should either vanish or leave a drop event.

	var itemWindow = this._activeItemWindow;

	// Just drops into the void
	if(HeldItemManager.isHoldingItem()){
		HeldItemManager.setHeldItem(null);
	} else {
		itemWindow.dropItem();
	}
	this.commandCancel();
}

Scene_InventoryBase.prototype.applyItem = function() {
	var action = new Game_Action(this.user());
	action.setItemObject(this.item());
	this.itemTargetActors().forEach(function(target) {
		for (var i = 0; i < action.numRepeats(); i++) {
			action.apply(target);
		}
	}, this);
	action.applyGlobal();
};

Scene_InventoryBase.prototype.user = function() {
	var members = $gameParty.movableMembers();
	var bestActor = members[0];
	var bestPha = 0;
	for (var i = 0; i < members.length; i++) {
		if (members[i].pha > bestPha) {
			bestPha = members[i].pha;
			bestActor = members[i];
		}
	}
	return bestActor;
};

Scene_InventoryBase.prototype.placeHeldItem = function() {
	var itemWindow = this._activeItemWindow;
	var heldItem = HeldItemManager.heldItem();
	var grid = itemWindow._data;
	var pos = grid.pos(itemWindow.index())
	if(grid.canPlaceHeldItem(heldItem, pos.col, pos.row)){
		var item = grid.placeHeldItem(heldItem, pos.col, pos.row);
		HeldItemManager.setHeldItem(item);
		if(TouchInput.isTriggered()) itemWindow.updateIndexMouse();
		itemWindow.refresh();
	} else {
		itemWindow.playBuzzerSound();
	}
}

Scene_InventoryBase.prototype.commandSwap = function() {
	this.placeHeldItem();
	this._heldCommandWindow.close();
	this.activateItemWindow();
}

Scene_InventoryBase.prototype.onItemOk = function(itemWindow) {
	this._activeItemWindow = itemWindow;
	this._activeItem = itemWindow.item();
	this._commandWindow.setItem(this._activeItem);
	this._heldCommandWindow.setItem(HeldItemManager.heldItem() ? HeldItemManager.heldItem().object() : null);
	this._heldCommandWindow.setTargetItem(this._activeItem);
	$gameParty.setLastItem(this.item());
	if(TouchInput._anyMousePressed){
		this.onItemOkMouse();
	} else {
		this.onItemOkKeyboard();
	}
};

Scene_InventoryBase.prototype.onItemOkMouse = function() {
	var itemWindow = this._activeItemWindow;
	// Moving Item
	if(HeldItemManager.isHoldingItem()){
		// Place/Swap Item
		if(TouchInput.isTriggered()){
			this.placeHeldItem();
		// Do Active Command
		} else {
			// Open Active Command Window
			this.openHeldCommandWindow();
		}
	// Pick up item to move
	} else if(TouchInput.isTriggered()){
		var gridItem = itemWindow.gridItem();
		itemWindow._data.moveItem(gridItem._index);
		HeldItemManager.setHeldItem(gridItem);
		itemWindow.refresh();
	} else if (TouchInput.isCancelled()) {
		// If is an item then immediately use.
		if(DataManager.isItem(this.item())){
			this.deactivateItemWindow();
			this.determineItem();
		// Else open the command window
		} else {
			this.openCommandWindow();
		}
	}
};

Scene_InventoryBase.prototype.openHeldCommandWindow = function(){
	var itemWindow = this._activeItemWindow;
	var gridItem = itemWindow.gridItem();
	var rect = itemWindow.itemRect(gridItem._index, gridItem);

	var x =   itemWindow._x +  itemWindow.standardPadding() + rect.width/2 + rect.x - this._heldCommandWindow.width/2;
	var y =   itemWindow._y +  itemWindow.standardPadding() + rect.y;
	y += y <  Graphics.boxHeight / 2 ? rect.height : -this._heldCommandWindow.height;
	
	x = x.clamp(0, Graphics.boxWidth - this._heldCommandWindow.width);
	y = y.clamp(0, Graphics.boxHeight - this._heldCommandWindow.height);

	this._heldCommandWindow.setPosition(x, y);
	this._heldCommandWindow.open();
	this._heldCommandWindow.activate();
	// Deactive the Player Inventory Item window
	this.deactivateItemWindow();
}

Scene_InventoryBase.prototype.openCommandWindow = function(){
	var itemWindow = this._activeItemWindow;
	var gridItem = itemWindow.gridItem();
	var rect = itemWindow.itemRect(gridItem._index, gridItem);

	var x =   itemWindow._x +  itemWindow.standardPadding() + rect.width/2 + rect.x - this._heldCommandWindow.width/2;
	var y =   itemWindow._y +  itemWindow.standardPadding() + rect.y;
	y += y <  Graphics.boxHeight / 2 ? rect.height : -this._heldCommandWindow.height;
	
	x = x.clamp(0, Graphics.boxWidth - this._heldCommandWindow.width);
	y = y.clamp(0, Graphics.boxHeight - this._heldCommandWindow.height);

	this._commandWindow.setPosition(x, y);
	this._commandWindow.open();
	this._commandWindow.activate();
	this.deactivateItemWindow();

}

Scene_InventoryBase.prototype.onItemOkKeyboard = function() {
	var itemWindow = this._activeItemWindow;
	var consumedEvent = false;
	// Moving Item
	if(HeldItemManager.isHoldingItem()){
		this.placeHeldItem();
		consumedEvent = true;
	// Pick up item to move
	} else {
		this.openCommandWindow(itemWindow);
		consumedEvent = true;
	}
	return consumedEvent;
};

Scene_InventoryBase.prototype.setActiveItem = function(item){
	this._activeItem = item;
}

Scene_ItemBase.prototype.activateItemWindow = function() {
    this._itemWindow.refresh();
	this._itemWindow.activate();
};

Scene_ItemBase.prototype.deactivateItemWindow = function() {
	this._itemWindow.deactivate();
};

Scene_InventoryBase.prototype.commandUse = function(){
	this._commandWindow.close();
	if(DataManager.isItem(this.item())){
		this.determineItem();
	} else {
		this.equipItem();
	}
}
Scene_InventoryBase.prototype.commandMove = function(){
	var itemWindow = this._activeItemWindow;
	this._commandWindow.close();
	this.activateItemWindow()
	this.moveItem();
	var pos = itemWindow.getRectPos();
	HeldItemManager.setPosition(pos.x, pos.y);
}
Scene_InventoryBase.prototype.commandDrop = function(){
	this._commandWindow.close();
	this._dropCommandWindow.open();
	this._dropCommandWindow.activate();
}
Scene_InventoryBase.prototype.commandCancel = function(){
	this._commandWindow.close();
	this._heldCommandWindow.close();
	this._dropCommandWindow.close();
	this.activateItemWindow()
}
Scene_InventoryBase.prototype.moveItem = function(){
	var itemWindow = this._activeItemWindow;
	var gridItem = itemWindow.gridItem();
	itemWindow._data.moveItem(gridItem._index);
	HeldItemManager.setHeldItem(gridItem);
	itemWindow.refresh();
}
Scene_InventoryBase.prototype.update = function() {
	Scene_ItemBase.prototype.update.call(this);
	var itemWindow = this._activeItemWindow;
	if(!this.isWithinWindows()){
		// Handle Dropping Item
		if(TouchInput.isTriggered() && HeldItemManager.isHoldingItem() && (Imported.YEP_X_ItemDisassemble ? !this._disassemblePoolWindow.isOpenAndActive() : true)){
			this._dropCommandWindow.open();
			this._dropCommandWindow.activate();
			this.deactivateItemWindow();
		}
		// Handle Closing Scene
		if(TouchInput.isCancelled() && !HeldItemManager.isHoldingItem()){
			SoundManager.playCancel();
			this.popScene();
		}
	}
	// Handle Showing HeldItem
	if(HeldItemManager.isHoldingItem()) {
		this._heldItemWindow.show();
	} else {
		this._heldItemWindow.hide();
	}	
}
Scene_InventoryBase.prototype.item = function() {
	return this._activeItem;
};
Scene_InventoryBase.prototype.playSeForItem = function() {
	SoundManager.playUseItem();
};
//-----------------------------------------------------------------------------
// Scene_ItemDrop
//
// The scene class of the grid item screen.
function Scene_ItemDrop() {
	this.initialize.apply(this, arguments);
}

MRP.GridInventory.Scene_ItemDrop = Scene_ItemDrop;
Scene_ItemDrop.prototype = Object.create(Scene_InventoryBase.prototype);
Scene_ItemDrop.prototype.constructor = Scene_ItemDrop;
Scene_ItemDrop.prototype.initialize = function() {
	Scene_InventoryBase.prototype.initialize.call(this);
	this._activeInventoryWindow = null;
	this._data = $gameTemp._itemDropGrid || new Game_Grid("Drop",0,0);
};
Scene_ItemDrop.prototype.update = function() {
	Scene_InventoryBase.prototype.update.call(this);
	if(Input.isTriggered("tab")){
		if(this._activeInventoryWindow == this._itemWindow){
			this.setKeyboardActiveInventoryWindow(this._dropWindow);
			this._dropWindow.select(this._itemWindow.index());
			this._itemWindow.select(-1);
		} else {
			this.setKeyboardActiveInventoryWindow(this._itemWindow);
			this._itemWindow.select(this._dropWindow.index());
			this._dropWindow.select(-1);
		}
	}
}

Scene_ItemDrop.prototype.setKeyboardActiveInventoryWindow = function(window) {
	if(this._activeInventoryWindow) this._activeInventoryWindow.deactivateKeyboard();
	else this._itemWindow.deactivateKeyboard();
	window.activateKeyboard();
	this._activeInventoryWindow = window;
};

Scene_ItemDrop.prototype.createInventoryWindows = function() {
	Scene_InventoryBase.prototype.createInventoryWindows.call(this);
	this.createDropWindow();
	this.createInformationSprite();
};

Scene_ItemDrop.prototype.createCommandWindow = function() {
	Scene_InventoryBase.prototype.createCommandWindow.call(this);
	this._commandWindow.removeUseCommand('USE');
}

Scene_ItemDrop.prototype.createToolTip = function() {
	Scene_InventoryBase.prototype.createToolTip.call(this);
	this._dropWindow.setToolTipWindow(this._toolTipWindow);
};

Scene_ItemDrop.prototype.createInformationSprite = function() {
	this._infoSprite = new Sprite();
	this._infoSprite.bitmap = new Bitmap(this._itemWindow.width, 48);
	
	// We treat the sprite as a window so it gets rendered in the correct order
	this._infoSprite._isWindow = true;
	this._infoSprite.openness = 1;

	var wy = Graphics.boxHeight/2 - 24;
	var wx = this._itemWindow.x;
	
	
	this._infoSprite.move(wx, wy);
	this._infoSprite.bitmap.drawText("Press 'TAB' to switch window", 0, 0, this._itemWindow.width, 48, 'center');

	this.addWindow(this._infoSprite);
};

Scene_ItemDrop.prototype.createDropWindow = function() {
	var wh = this._itemWindow.height;
	var ww = this._itemWindow.width;
	var wy = 0;
	var wx = this._itemWindow.x;
	this._dropWindow = new Window_ItemGrid(wx, wy, ww, wh);
	this._dropWindow.setHandler('ok',     this.onItemOk.bind(this, this._dropWindow));
	this._dropWindow.setHandler('cancel', this.popScene.bind(this));
	this.addWindow(this._dropWindow);
	
	this.setKeyboardActiveInventoryWindow(this._dropWindow);
	this._dropWindow.setData(this._data);
	this._dropWindow.activate();
};

Scene_ItemDrop.prototype.isWithinWindows = function(){
	if(this._dropWindow.isTouchedInsideFrame()){
		return true;
	}
	return Scene_InventoryBase.prototype.isWithinWindows.call(this);
}

Scene_ItemDrop.prototype.openHeldCommandWindow = function(){
	Scene_InventoryBase.prototype.openHeldCommandWindow.call(this);
	this._dropWindow.deactivate();
}

Scene_ItemDrop.prototype.activateItemWindow = function() {
	Scene_ItemBase.prototype.activateItemWindow.call(this);
    this._dropWindow.refresh();
	this._dropWindow.activate();
};

Scene_ItemDrop.prototype.deactivateItemWindow = function() {
	Scene_ItemBase.prototype.deactivateItemWindow.call(this);
	this._dropWindow.deactivate();
};
//-----------------------------------------------------------------------------
// Scene_Inventory
//
// The scene class of the grid item screen.
function Scene_Inventory() {
	this.initialize.apply(this, arguments);
}

MRP.GridInventory.Scene_Inventory = Scene_Inventory;
Scene_Inventory.prototype = Object.create(Scene_InventoryBase.prototype);
Scene_Inventory.prototype.constructor = Scene_Inventory;
Scene_Inventory.prototype.initialize = function() {
	Scene_InventoryBase.prototype.initialize.call(this);
	this._equipWindowIndex = 0;
	this._slotIndex = 0;
	this._equipSelection = false;
};

Scene_Inventory.prototype.update = function() {
	Scene_InventoryBase.prototype.update.call(this);
	
	var x = TouchInput.x;
	var y = TouchInput.y;
	
	// Handle Mouse Moving Between Equip Windows
	if(TouchInput.isTriggered() && y < Graphics.boxHeight - 280){
		if(x < 18) this.equipWindowLeft();
		else if(x > Graphics.boxWidth - 18) this.equipWindowRight();
	}

	if(this._equipSelection){
		if(Input.isRepeated('right')) {
			this.goToNextEquipWindow();
		}
		if(Input.isRepeated("left")) {
			this.goToPrevEquipWindow();
		}
		if(Input.isRepeated('up')){
			this.goToPrevEquipSlot();
		}
		if(Input.isRepeated('down')){
			this.goToNextEquipSlot();
		}
		if(Input.isTriggered("ok")) {
			var window = this._equipWindows[this._equipWindowIndex];
			var slot = window._equipSlots[this._slotIndex];
			if(HeldItemManager.isHoldingItem()){
				if(slot.equipHeldItem()) {
					this.exitEquipSelection();				
				}
			} else {
				slot.unEquipItem();
				this._activeItemWindow.select(0);
			}
		}
		if(Input.isTriggered("cancel")) {
			this.exitEquipSelection();
		}
	}
	// Tab between equipMode and Normal mode
	if(Input.isTriggered("tab")){
		this._equipSelection = !this._equipSelection;
		if(this._equipSelection){
			this._activeItemWindow.deactivateKeyboard();
			this._activeItemWindow.select(-1);

			this.initialEquipWindow();
		} else {
			this.highlightEquipWindow(false);
			this._activeItemWindow.activateKeyboard();
			this._activeItemWindow.select(0);
		}
	}
}

Scene_Inventory.prototype.createInventoryWindows = function() {
	Scene_InventoryBase.prototype.createInventoryWindows.call(this);
	this._equipWindows = [];
	this.createEquipWindows();
};

Scene_Inventory.prototype.createToolTip = function() {
	Scene_InventoryBase.prototype.createToolTip.call(this);
	for(var i = 0; i < this._equipWindows.length; i++){
		this._equipWindows[i].setToolTipWindow(this._toolTipWindow);
	}
};

Scene_Inventory.prototype.createEquipWindows = function() {
	var w = Graphics.boxWidth / MRP.GridInventory.numberOfEquipmentWindows;
	var partyMembers = MRP.GridInventory.onlyShowBattleMembers ? $gameParty.battleMembers() : $gameParty.members();
	
	for(var i = 0; i < partyMembers.length; i++){
		var window = new Window_Equip(0, 0, w, Graphics.boxHeight - 280);
		window.setHandler('onEquip', this.onEquip.bind(this));
		window.setHandler('cancel', this.popScene.bind(this));
		window.setActor(partyMembers[i]);
		this._equipWindows.push(window);
		this.addWindow(window);
	}
	
	this._equipIndex = 0;
	this.updateEquipWindowsPositions();
};

Scene_Inventory.prototype.equipWindowLeft = function(){
	this._equipIndex = Math.max(this._equipIndex - 1, 0);
	this.updateEquipWindowsPositions();
	
}

Scene_Inventory.prototype.equipWindowRight = function(){
	var partyMembers = MRP.GridInventory.onlyShowBattleMembers ? $gameParty.battleMembers() : $gameParty.members();
	this._equipIndex = Math.min(this._equipIndex + 1, partyMembers.length - MRP.GridInventory.numberOfEquipmentWindows);
	this.updateEquipWindowsPositions();
}

Scene_Inventory.prototype.updateEquipWindowsPositions = function(){
	var partyMembers = MRP.GridInventory.onlyShowBattleMembers ? $gameParty.battleMembers() : $gameParty.members();
	var w = Graphics.boxWidth / MRP.GridInventory.numberOfEquipmentWindows;
	var baseX = MRP.GridInventory.numberOfEquipmentWindows < this._equipWindows.length ? -w * this._equipIndex : (Graphics.boxWidth - w *  partyMembers.length) / 2;;
	for(var i = 0; i < this._equipWindows.length; i++){
		var newX = (this._equipWindows[i-1] ? this._equipWindows[i-1].x  + this._equipWindows[i-1].width : baseX);
		var window = this._equipWindows[i];

		if(newX < 0 || newX > Graphics.boxWidth) window.hide();
		else window.show();

		window.move(newX, window.y, w, window.height);
		window.resetArrowVisibility();
	}

	if(this._equipIndex > 0) this._equipWindows[this._equipIndex].showLeftArrow(true);
	if(this._equipIndex + MRP.GridInventory.numberOfEquipmentWindows < this._equipWindows.length) this._equipWindows[this._equipIndex + MRP.GridInventory.numberOfEquipmentWindows - 1].showRightArrow(true);

}

Scene_Inventory.prototype.isWithinWindows = function(){
	if(!Scene_InventoryBase.prototype.isWithinWindows.call(this)){
		for(var i = 0; i < this._equipWindows.length; i++){
			var window = this._equipWindows[i];
			if(window.isOpen() && window.isTouchedInsideFrame()){
				return true;
			}
		}
		return false;
	}
	return true;
}

Scene_Inventory.prototype.showSubWindow = function(window) {
    Scene_InventoryBase.prototype.showSubWindow.call(this, window);
    this.deactivateEquipWindows();
};


Scene_Inventory.prototype.hideSubWindow = function(window) {
	Scene_InventoryBase.prototype.hideSubWindow.call(this, window);
    this.activateEquipWindows();
};

Scene_Inventory.prototype.activateEquipWindows = function() {
    for(var i = 0; i < this._equipWindows.length; i++){
		this._equipWindows[i].activate();
	}
};

Scene_Inventory.prototype.deactivateEquipWindows = function() {
    for(var i = 0; i < this._equipWindows.length; i++){
		this._equipWindows[i].deactivate();
	}
};

Scene_Inventory.prototype.onEquip = function() {
	this.exitEquipSelection();
	this._itemWindow.refresh();
}

Scene_Inventory.prototype.isWithinInEquipWindows = function(){
	// Can collapse these if statements
	for(var i = 0; i < this._equipWindows.length; i++){
		if(this._equipWindows[i].isTouchedInsideFrame()){
			return true;
		}
	}
	return false;
}

Scene_Inventory.prototype.equipItem = function(){
	var itemWindow = this._activeItemWindow;
	this._equipSelection = true;
	this.commandMove();
	itemWindow.deactivateKeyboard();
	itemWindow.select(-1);
	this.initialEquipWindow();
}

Scene_Inventory.prototype.highlightEquipWindow = function(value) {
	this._equipWindows[this._equipWindowIndex].setHighlight(value);
	this._equipWindows[this._equipWindowIndex]._equipSlots[this._slotIndex].setHighlight(value);
}

Scene_Inventory.prototype.exitEquipSelection = function(){
	this.highlightEquipWindow(false);
	this._equipSelection = false;
	this._activeItemWindow.activateKeyboard();
	this._activeItemWindow.select(0);
}

Scene_Inventory.prototype.ensureEquipWindowVisible = function() {
	if(this._equipWindowIndex < this._equipIndex){
		this.equipWindowLeft();
		this.ensureEquipWindowVisible();
	} else if(this._equipWindowIndex > this._equipIndex + MRP.GridInventory.numberOfEquipmentWindows - 1){
		this.equipWindowRight();
		this.ensureEquipWindowVisible();
	}
}

Scene_Inventory.prototype.initialEquipWindow = function(){
	this.highlightEquipWindow(false);
	if(HeldItemManager.isHoldingItem()){
		var heldItem = HeldItemManager.heldItem();
		// Search the windows that can be seen now
		this._equipWindowIndex = this.getNextEquipWindow(this._equipIndex, heldItem.object());
		if(this._equipWindowIndex != null) {
			var equipWindow = this._equipWindows[this._equipWindowIndex];
			this._slotIndex = equipWindow.getEquipSlotIDs(heldItem.object().etypeId)[0];
			var slot = equipWindow._equipSlots[this._slotIndex];
			this.ensureEquipWindowVisible();
			HeldItemManager.setPosition(equipWindow.x + slot.x, equipWindow.y + slot.y);

		} else {
			return false;
		}
	} else {
		this._equipWindowIndex = this._equipIndex;
		this._slotIndex = 0;
		var equipWindow = this._equipWindows[this._equipWindowIndex];
		var slot = equipWindow._equipSlots[this._slotIndex];
		HeldItemManager.setPosition(equipWindow.x + slot.x, equipWindow.y + slot.y);
	}
	this.highlightEquipWindow(true);
	return true;
}

Scene_Inventory.prototype.goToNextEquipSlot = function(){
	this.highlightEquipWindow(false);

	var slot = this.getNextEquipSlot();
	if(slot == null) return;

	var window = this._equipWindows[this._equipWindowIndex];
	this.highlightEquipWindow(true);

	this.ensureEquipWindowVisible();
	HeldItemManager.setPosition(window.x + slot.x, window.y + slot.y);
	SoundManager.playCursor();
}


Scene_Inventory.prototype.goToPrevEquipSlot = function(){
	this.highlightEquipWindow(false);

	var slot = this.getPrevEquipSlot();
	if(slot == null) return;

	var window = this._equipWindows[this._equipWindowIndex];
	this.highlightEquipWindow(true);

	this.ensureEquipWindowVisible();
	HeldItemManager.setPosition(window.x + slot.x, window.y + slot.y);
	SoundManager.playCursor();
}

Scene_Inventory.prototype.goToNextEquipWindow = function(){
	this.highlightEquipWindow(false);

	// Update Window
	this._equipWindowIndex = this.getNextEquipWindow(this._equipWindowIndex + 1, HeldItemManager.isHoldingItem() ? HeldItemManager.heldItem().object() : undefined);
	var window = this._equipWindows[this._equipWindowIndex];
	
	// Update Slot
	this._slotIndex = HeldItemManager.isHoldingItem() ? window.getEquipSlotIDs(HeldItemManager.heldItem().object().etypeId)[0] : 0;
	var slot = window._equipSlots[this._slotIndex];

	this.highlightEquipWindow(true);
	this.ensureEquipWindowVisible();

	HeldItemManager.setPosition(window.x + slot.x, window.y + slot.y);
	SoundManager.playCursor();
}

Scene_Inventory.prototype.goToPrevEquipWindow = function(){
	this.highlightEquipWindow(false);

	// Update Window
	this._equipWindowIndex = this.getPrevEquipWindow(this._equipWindowIndex - 1, HeldItemManager.isHoldingItem() ? HeldItemManager.heldItem().object() : undefined);
	var window = this._equipWindows[this._equipWindowIndex];
	
	// Update Slot
	this._slotIndex = HeldItemManager.isHoldingItem() ? window.getEquipSlotIDs(HeldItemManager.heldItem().object().etypeId)[0] : 0;
	var slot = window._equipSlots[this._slotIndex];

	this.highlightEquipWindow(true);
	this.ensureEquipWindowVisible();

	HeldItemManager.setPosition(window.x + slot.x, window.y + slot.y);
	SoundManager.playCursor();
}

Scene_Inventory.prototype.getPrevEquipSlot = function(){
	var item = HeldItemManager.heldItem();
	var equipWindow = this._equipWindows[this._equipWindowIndex];
	this._slotIndex = item ? equipWindow.getEquipSlotIDs(item.object().etypeId, 0, this._slotIndex).pop() : this._slotIndex - 1;
	if(this._slotIndex == undefined || this._slotIndex < 0) {
		//this._equipWindowIndex = (--this._equipWindowIndex).clamp(0, this._equipWindows.length - 1);
		this._equipWindowIndex = this.getPrevEquipWindow(this._equipWindowIndex - 1, item ? item.object() : undefined);
		this._slotIndex = item ? this._equipWindows[this._equipWindowIndex].getEquipSlotIDs(item.object().etypeId).pop() : this._equipWindows[this._equipWindowIndex]._equipSlots.length - 1;
	}
	equipWindow = this._equipWindows[this._equipWindowIndex];
	var slots = equipWindow._equipSlots;
	return slots[this._slotIndex];
}

Scene_Inventory.prototype.getNextEquipSlot = function(){
	var item = HeldItemManager.heldItem();
	var equipWindow = this._equipWindows[this._equipWindowIndex];
	this._slotIndex = item ? equipWindow.getEquipSlotIDs(item.object().etypeId, this._slotIndex + 1)[0] : this._slotIndex + 1;
	if(this._slotIndex == undefined || this._slotIndex >= equipWindow._equipSlots.length) {
		this._equipWindowIndex = this.getNextEquipWindow(this._equipWindowIndex + 1, item ? item.object() : undefined);
		if(this._equipWindowIndex == null) {
			this._equipWindowIndex = 0;
			this._slotIndex = 0;
			return null;
		}
		this._slotIndex =  item ? equipWindow.getEquipSlotIDs(item.object().etypeId)[0] : 0;
	}
	equipWindow = this._equipWindows[this._equipWindowIndex];
	slots = equipWindow._equipSlots;
	return slots[this._slotIndex];
}

// Search to the right
Scene_Inventory.prototype.getNextEquipWindow = function(index, item) {
	if(item != undefined){
		for(var i = index; i < this._equipWindows.length; i++){
			var window = this._equipWindows[i];
			var slots = window.getEquipSlotIDs(item.etypeId);
			if(slots.length > 0) return i;
		}
		for(var i = 0; i < index; i++){
			var window = this._equipWindows[i];
			var slots = window.getEquipSlotIDs(item.etypeId);
			if(slots.length > 0) return i;
		}
		return null;
	} else {
		return index < this._equipWindows.length ? index : 0;
	}
}

// Search to the left
Scene_Inventory.prototype.getPrevEquipWindow = function(index, item) {
	if(item != undefined){
		for(var i = index; i >= 0; i--){
			var window = this._equipWindows[i];
			var slots = window.getEquipSlotIDs(item.etypeId);
			if(slots.length > 0) return i;
		}
		for(var i = this._equipWindows.length - 1; i > index; i--){
			var window = this._equipWindows[i];
			var slots = window.getEquipSlotIDs(item.etypeId);
			if(slots.length > 0) return i;
		}
		return null;
	} else {
		return index >= 0 ? index : this._equipWindows.length - 1;
	}
}

//-----------------------------------------------------------------------------
// Scene_Menu
//
// Chagnes to Scene_Menu.

Scene_Menu.prototype.commandItem = function() {
    SceneManager.push(Scene_Inventory);
};

//-----------------------------------------------------------------------------
// Game_Grid
//
// The game object class for a picture.

function Game_Grid() {
	this.initialize.apply(this, arguments);
}
Game_Grid.prototype.initialize = function(name, cols, rows, storeable = true) {
	this.initBasic(name, cols, rows, storeable);
	this.initGrid();
};
Game_Grid.prototype.initBasic = function(name, cols, rows, storeable) {
	this._name = name;
	this._cols = cols;
	this._rows = rows;
	this._grid = [];
	this._storeable = storeable;
};

Game_Grid.prototype.initGrid = function() {
	this._grid = [];
	for(var i = 0; i < this.cols(); i++){
		this._grid.push([]);
		for(var j = 0; j < this.rows(); j++){
			this._grid[i].push(this.createEmptyItem(this.index(i,j)));
		}
	}
};

Game_Grid.prototype.isEmpty = function(){
	for(var i = 0; i < this.cols(); i++){
		for(var j = 0; j < this.rows(); j++){
			if(!this._grid[i][j].isEmpty()) return false;
		}
	}
	return true;
}

Game_Grid.prototype.cols = function(){
	return this._cols;
};

Game_Grid.prototype.rows = function(){
	return this._rows;
};
Game_Grid.prototype.gridItem = function(index){
	var pos = this.pos(index);
	return this._grid[pos.col][pos.row];
}

Game_Grid.prototype.createEmptyItem = function(index){
	return new Game_GridItem(index);
}

Game_Grid.prototype.createItem = function(index, item){
	return new Game_GridItem(index, item);
}
Game_Grid.prototype.index = function(col, row) {
	return Math.max(-1, this._cols * row + col);
}
Game_Grid.prototype.pos = function(index){
	var col = index % this._cols;
	var row = Math.floor(index / this._cols);
	return {col: col, row: row};
}

Game_Grid.prototype.pushItem = function(item, amount){
	if(!this.includes(item)) return;
	amount = this.tryPlaceOwnedItem(item, amount);
	if(amount > 0) {
		amount = this.tryPlaceItem(item, amount);
	}
	if(amount > 0){
		// HANDLE WHAT SHOULD HAPPEN IF NO FREE SPACE
		var items = [];
		for(var i = 0; i < amount; i++) items[i] = item;
		$gameMap.makeDropEvent(items);
		//$gameMap._events.push(new Game_DropEvent($gameMap.events().length + 1, $gameMap._mapId, items));
	}
}

Game_Grid.prototype.includes = function(item) {
	if (DataManager.isItem(item)) {
		if(item.itypeId == 1 || item.itypeId == 2) return true;
	} else if (DataManager.isWeapon(item)) {
		return true;
	} else if (DataManager.isArmor(item)) {
		return true;
	}
	return false;
};

Game_Grid.prototype.tryPlaceOwnedItem = function(item, amount){
	for(var col = 0; col < this.cols(); col++){
		for(var row = 0; row < this.rows(); row++){
			if(this.canPlaceOwnedItem(item, col, row)){
				amount = this.placeOwnedItem(amount, col, row);
				if(amount == 0) return amount;
			}
		}
	}
	return amount;
}

Game_Grid.prototype.placeOwnedItem = function(amount, col, row) {
	var gridItem = this._grid[col][row];
	gridItem = gridItem._parent != null ? this.gridItem(gridItem._parent) : gridItem;
	
	// Update amount
	var oldAmount = gridItem.numItems();
	gridItem._amount = Math.min(oldAmount + amount, gridItem.maxItems());
	
	var gain = amount > 0;
	
	var diff = (gridItem.maxItems() - oldAmount);
	amount = gain ? amount - diff : amount + oldAmount;

	// Clean up
	if(gridItem.numItems() <= 0) {
		for(var i = 0; i < gridItem._children.length; i++){
			var childIndex = gridItem._children[i];
			var pos = this.pos(childIndex);
			this._grid[pos.col][pos.row] = this.createEmptyItem(childIndex);
		}
		this._grid[col][row] = this.createEmptyItem(gridItem._index);
	}
	return gain ? Math.max(amount, 0) : Math.min(amount, 0);
}

// Only looks at occupied spots.
Game_Grid.prototype.canPlaceOwnedItem = function(item, col, row) {
	var gridItem = this._grid[col][row];
	gridItem = gridItem._parent != null ? this.gridItem(gridItem._parent) : gridItem;
	return gridItem.isSame(item);
}

Game_Grid.prototype.tryPlaceItem = function(item, amount) {
	for(var col = 0; col < this.cols(); col++){
		for(var row = 0; row < this.rows(); row++){
			if(this.canPlaceItem(item, col, row)){
				amount = this.placeItem(item, amount, col, row);
				if(amount == 0) return amount;
			}
		}
	}
	return amount;
}

// Only looks at empty spots.
Game_Grid.prototype.canPlaceItem = function(item, col, row) {
	var index = this.index(col, row);
	var gridItem = this._grid[col][row];
	var newItem = this.createItem(index, item);
	if(gridItem.isEmpty()){
		for(var i = 0; i < newItem._width; i++){
			for(var j = 0; j < newItem._height; j++){
				if((i == 0 && j == 0)) continue;
				if(!this.isWithinGrid(col+i, row+j)) return false;
				var spot = this._grid[col+i][row+j];
				if(!spot.isEmpty()) return false;
			}
		}
		return true;
	}
	return false;
}

// Place should be empty & amount should be postive. To remove an item use placeOwnedItem with a negatative
Game_Grid.prototype.placeItem = function(item, amount, col, row) {
	var index = this.index(col, row);
	var gridItem = this.createItem(index, item);
	this._grid[col][row] = gridItem;

	// Place children
	for(var i = 0; i <  gridItem._width; i++){
		for(var j = 0; j < gridItem._height; j++){
			if(i == 0 && j == 0) continue;
			var childIndex = this.index(col+i, row+j);
			var childItem = this.createItem(childIndex, item);
			childItem._parent = index;
			gridItem._children.push(childIndex);
			this._grid[col+i][row+j] = childItem;
		}
	}
	
	// Update amount
	gridItem._amount = Math.min(amount, gridItem.maxItems());
	amount = amount - gridItem.maxItems();
	
	return Math.max(amount, 0);
}


// Asumes you can place (use pushHeldItem if you are not sure)
Game_Grid.prototype.placeHeldItem = function(gridItem, col, row){		
	var replacedItem = this._grid[col][row];
	if(replacedItem._parent == null && replacedItem.isSame(gridItem.object()) && replacedItem.numItems() + gridItem.numItems() <= replacedItem.maxItems()){
		replacedItem._amount += gridItem._amount;
		return null;
	}
	
	if(replacedItem._parent != null) {
		var pos = this.pos(replacedItem._parent);
		replacedItem = this._grid[pos.col][pos.row];
	}
	this.removeItem(col, row);
	gridItem._children = [];
	gridItem._index = this.index(col, row);
	// Place the item
	this._grid[col][row] = gridItem;
	for(var i = 0; i <  gridItem._width; i++){
		for(var j = 0; j < gridItem._height; j++){
			if(i == 0 && j == 0) continue;
			var spot = this._grid[col+i][row+j];
			if(!spot.isEmpty()){
				if(spot._parent != null) {
					var pos = this.pos(spot._parent);
					replacedItem = this._grid[pos.col][pos.row];
					this.removeItem(pos.col, pos.row);
				} else {
					replacedItem = spot;
					this.removeItem(col+i, row+j);
				}
			} 
			var childIndex = this.index(col+i, row+j);
			var childItem = this.createItem(childIndex, gridItem.object());
			childItem._parent = gridItem._index;
			gridItem._children.push(childIndex);
			this._grid[col+i][row+j] = childItem;
		}
	}
	
	if(replacedItem.isEmpty()) {
		return null;
	} else {
		return replacedItem;
	}
}

Game_Grid.prototype.storeable = function() {
	return this._storeable;
}

Game_Grid.prototype.canPlaceHeldItem = function(heldItem, col, row){
	if(!this.storeable()) return false;
	var swapItem = null;		
	for(var i = 0; i < heldItem._width; i++){
		for(var j = 0; j < heldItem._height; j++){
			if(!this.isWithinGrid(col+i, row+j)) return false;
			var spot = this._grid[col+i][row+j];
			spot = spot._parent != null ? this.gridItem(spot._parent) : spot;
			if(!spot.isEmpty()) {
				if(swapItem != null && swapItem != spot._index) return false;
				else swapItem =  spot._index;
			}
		}
	}	
	return swapItem ? this.gridItem(swapItem) : true;
}

Game_Grid.prototype.isWithinGrid = function(col, row){
	return col > -1 && col < this._grid.length && row < this._grid[col].length;
}

Game_Grid.prototype.removeItem = function(col, row) {
	var gridItem = this._grid[col][row];
	if(gridItem._parent != null) {
		var pos = this.pos(gridItem._parent);
		gridItem = this._grid[pos.col][pos.row];
		col = pos.col;
		row = pos.row;
	}
	for(var i = 0; i < gridItem._width; i++){
		for(var j = 0; j < gridItem._height; j++){
			this._grid[col+i][row+j] = this.createEmptyItem(this.index(col+i, row+j));
		}
	}
}

Game_Grid.prototype.numItems = function(item){
	var amount = 0;
	for(var col = 0; col < this.cols(); col++){
		for(var row = 0; row < this.rows(); row++){
			var gridItem = this._grid[col][row];
			if(gridItem.isSame(item)){
				amount += gridItem._amount;
			}
		}
	}
	return amount;
}
Game_Grid.prototype.parentIndex = function(index) {
	var pos = this.pos(index);
	return this._grid[pos.col][pos.row]._parent;
}
// TODO:
Game_Grid.prototype.moveItem = function(index) {
	var pos = this.pos(index);
	var gridItem = this._grid[pos.col][pos.row];
	this.removeItem(pos.col, pos.row);
}
//-----------------------------------------------------------------------------
// Game_PlayerGrid
//
// The game object class for a the player inventory

function Game_PlayerGrid() {
	this.initialize.apply(this, arguments);
}
Game_PlayerGrid.prototype = Object.create(Game_Grid.prototype);
Game_PlayerGrid.prototype.constructor = Game_PlayerGrid;
Game_PlayerGrid.prototype.initialize = function(name, cols, rows) {
	Game_Grid.prototype.initialize.call(this, name, cols, rows);
};

// Used to help update the normal inventory container (for compatibility and to easily hasItem)
Game_PlayerGrid.prototype.updateContainer = function(item, amount){
	var container = $gameParty.itemContainer(item);
	var lastNumber = container[item.id] ? container[item.id] : 0;
	container[item.id] = lastNumber + amount;
	if (container[item.id] === 0) {
		delete container[item.id];
	}
}
Game_PlayerGrid.prototype.placeHeldItem = function(gridItem, col, row){
	if(!gridItem.isEmpty()) this.updateContainer(gridItem.object(), gridItem.numItems());
	return Game_Grid.prototype.placeHeldItem.call(this, gridItem, col, row);
}
Game_PlayerGrid.prototype.removeItem = function(col, row) {
	var gridItem = this._grid[col][row];
	if(gridItem._parent != null) {
		var pos = this.pos(gridItem._parent);
		gridItem = this._grid[pos.col][pos.row];
		col = pos.col;
		row = pos.row;
	}
	
	if(!gridItem.isEmpty()) this.updateContainer(gridItem.object(), -gridItem.numItems());
	for(var i = 0; i < gridItem._width; i++){
		for(var j = 0; j < gridItem._height; j++){
			this._grid[col+i][row+j] = this.createEmptyItem(this.index(col+i, row+j));
		}
	}
}

//-----------------------------------------------------------------------------
// Game_GridItem
//
// The game object class for a Grid Item.

function Game_GridItem() {
	this.initialize.apply(this, arguments);
}
Game_GridItem.prototype.initialize = function(index, item, amount) {
	this.initBasic(index, item, amount);
};

Game_GridItem.prototype.initBasic = function(index, item, amount) {
	this._id = item ? item.id : 0;
	this._dataClass = item ? this.getDataClass(item) : "";
	this._amount = amount ? amount : 0;
	this._parent = null;
	this._children = [];
	this.iconIndex = item ? item.meta.iconImage : null;
	
	// How many spaces it should take up
	var gridSizeString = item ? item.meta.GridSize : undefined;
	if(gridSizeString){
		var matches = (/(\d+),(\d+)/g).exec(gridSizeString);
		this._width = Number(matches[1]);
		this._height = Number(matches[2]);
	} else {	
		this._width = 1;
		this._height = 1;
	}
	this._maxItems = DataManager.getMaxItems(item);
	this._index = index;
};

Game_GridItem.prototype.isSame = function(item) {
	if(Imported.YEP_ItemCore){
		var baseItem = DataManager.getBaseItem(item);
		return (this._dataClass == this.getDataClass(baseItem) && this._id == item.id)
	}
	return (this._dataClass == this.getDataClass(item) && this._id == item.id)
}

Game_GridItem.prototype.numItems = function() {
	return this._amount;
}

Game_GridItem.prototype.object = function() {
	if (this.isItem()) {
		return $dataItems[this._id];
	} else if (this.isWeapon()) {
		return $dataWeapons[this._id];
	} else if (this.isArmor()) {
		return $dataArmors[this._id];
	} else {
		return null;
	}
};

Game_GridItem.prototype.maxItems = function() {
	return this._maxItems;
};

Game_GridItem.prototype.isItem = function() {
	return this._dataClass === 'item';
};
Game_GridItem.prototype.isEmpty = function() {
	return this._dataClass === '';
};
Game_GridItem.prototype.isWeapon = function() {
	return this._dataClass === 'weapon';
};
Game_GridItem.prototype.isArmor = function() {
	return this._dataClass === 'armor';
};

Game_GridItem.prototype.isEnabled = function() {
	return $gameParty.canUse(this.object()) || $gameParty.canEquip(this.object());
};
Game_GridItem.prototype.isParentOf = function(childIndex) {
	for(var i = 0; i < this._children.length; i++){
		if(childIndex == this._children[i]) return true;
	}
	return false;
}
	
Game_GridItem.prototype.getDataClass = function(item) {
	if (DataManager.isSkill(item)) {
		return 'skill';
	} else if (DataManager.isItem(item)) {
		return 'item';
	} else if (DataManager.isWeapon(item)) {
		return 'weapon';
	} else if (DataManager.isArmor(item)) {
		return 'armor';
	} else {
		return '';
	}
};
//-----------------------------------------------------------------------------
// Game_DropEvent
//
// The game object class for a Grid Item.

function Game_DropEvent() {
	this.initialize.apply(this, arguments);
}
Game_DropEvent.prototype = Object.create(Game_Event.prototype);

Game_DropEvent.prototype.constructor = Game_DropEvent;

Game_DropEvent.prototype.initialize = function(eventId, mapId, items) {		
	this._inventoryId = "E" + eventId + "M" + mapId;
	this._eventId = eventId;
	this._eventX = $gamePlayer.x;
	this._eventY = $gamePlayer.y;

	DataManager.makeDropInventory(this._inventoryId, MRP.GridInventory.playerCols, MRP.GridInventory.playerRows);
	this.pushItems(items);


	Game_Event.prototype.initialize.call(this, mapId, eventId);
};

Game_DropEvent.prototype.pushItems = function(items) {
	var inventory = this.inventory();
	for(var i = 0; i < items.length; i++) {
		inventory.pushItem(items[i], 1);
	}
}

Game_DropEvent.prototype.inventory = function() {
	return MRP.GridInventory.dropInventories[this._inventoryId];
}

Game_DropEvent.prototype.event = function() {
	return MRP.GridInventory.makeEventData(this._eventId, this._inventoryId, this._eventX, this._eventY);
};

//-----------------------------------------------------------------------------
// Game_Map
//
// Changes to Game_Map

MRP.GridInventory.Game_Map_setup = Game_Map.prototype.setup;
Game_Map.prototype.setup = function(mapId) {
	MRP.GridInventory.dropInventories = {};
	MRP.GridInventory.Game_Map_setup.call(this, mapId);
};


//-----------------------------------------------------------------------------
// HeldItemManager
//
// A manager which handles the held item.

function HeldItemManager() {
    throw new Error('This is a static class');
}

HeldItemManager._heldItem = null;
HeldItemManager._x = 0; // Where the held item should be x-axis
HeldItemManager._y = 0; // Where the held item should be y-axis 

HeldItemManager.setHeldItem = function(item) {
	this._heldItem = item;
}

HeldItemManager.heldItem = function() {
	return this._heldItem;
}

HeldItemManager.isHoldingItem = function() {
	return this._heldItem != null;
}

HeldItemManager.loseHeldItem = function(amount) {
	if(this.isHoldingItem()){
		var item = this._heldItem.object();
		if(DataManager.isItem(item)){
			this._heldItem._amount -= amount;
			if(this._heldItem.numItems() <= 0){
				this._heldItem = null;
			}
		}
	}
}

HeldItemManager.getPosition = function() {
	return {"x":this._x, "y":this._y};
}

HeldItemManager.setPosition = function(x, y) {
	this._x = x;
	this._y = y;
}

//-----------------------------------------------------------------------------
// DataManager
//
// Changes to DataManager

DataManager._databaseFiles.push(
	{ name: '$dataInventoryIcons', src: 'InventoryIcons.json' }
);

MRP.GridInventory.DataManager_isDatabaseLoaded = DataManager.isDatabaseLoaded;
DataManager.isDatabaseLoaded = function() {
  	if (!MRP.GridInventory.DataManager_isDatabaseLoaded.call(this)) return false;
  	if (!MRP._loaded_MRP_GridInventory) {
		this.processInventories();
		this.processToolTipNotetags1($dataItems);
		this.processToolTipNotetags1($dataWeapons); 
		this.processToolTipNotetags1($dataArmors);
		this.processToolTipNotetags2($dataItems);
		this.processToolTipNotetags2($dataWeapons); 
		this.processToolTipNotetags2($dataArmors);
		MRP._loaded_MRP_GridInventory = true;
  	}
  	return true;
};

DataManager.processToolTipNotetags1 = function(group){
	var headerR = /<ToolTipVisibility: *(.+) *, *(.+)>/i;
	var headerMultiR = /<ToolTipVisibility>/i;
	var tagR = /(.+):(.+)/i;
	var tailR = /<\/ToolTipVisibility>/i;
	  
	for(var i = 1; i < group.length; i++){
		var obj = group[i];
		var lines = obj.note.split(/\n+/)
		
		obj.toolTipVisibility = []
		var inTag = false;

		for(var j = 0; j < lines.length; j++) {
			var line = lines[j];
			if(line.match(headerR) || (inTag && line.match(tagR))){
				var id = RegExp.$1;
				var tag = RegExp.$2;
				obj.toolTipVisibility[id.trim()] = (tag.toLowerCase().trim() == "true")
			} else if(line.match(headerMultiR)){
				inTag = true;
			} else if (line.match(tailR)){
				inTag = false;
			}
		}
	}
}

DataManager.processToolTipNotetags2 = function(group){
	var headerR = /<ToolTipFormat: *(.+) *, *(.+)>/i;
	var headerMultiR = /<ToolTipFormat>/i;
	var tagR = /(.+):(.+)/i;
	var tailR = /<\/ToolTipFormat>/i;
	  
	for(var i = 1; i < group.length; i++){
		var obj = group[i];
		var lines = obj.note.split(/\n+/)
		
		obj.toolTipFormat = []
		var inTag = false;

		for(var j = 0; j < lines.length; j++) {
			var line = lines[j];
			if(line.match(headerR) || (inTag && line.match(tagR))){
				var id = RegExp.$1;
				var tag = RegExp.$2;
				obj.toolTipFormat[id.trim()] = tag;
			} else if(line.match(headerMultiR)){
				inTag = true;
			} else if (line.match(tailR)){
				inTag = false;
			}
		}
	}
}

DataManager.processInventories = function(){
	MRP.GridInventory.inventories = MRP.GridInventory.dataInventories.map(function(e){
		var inventory = new Game_Grid(e.Name, e.Columns, e.Rows);
		for(var i = 0; i < e.Weapons.length; i++) {
			inventory.pushItem($dataWeapons[e.Weapons[i]], 1);
		}
		for(var i = 0; i < e.Armors.length; i++) {
			inventory.pushItem($dataArmors[e.Armors[i]], 1);
		}
		for(var i = 0; i < e.Items.length; i++) {
			inventory.pushItem($dataItems[e.Items[i]], 1);
		}
		return inventory;
	});
}

MRP.GridInventory.DataManager_makeSaveContents = DataManager.makeSaveContents;
DataManager.makeSaveContents = function() {
    // A save data does not contain $gameTemp, $gameMessage, and $gameTroop.
    var contents = MRP.GridInventory.DataManager_makeSaveContents.call(this);
	contents.inventories       = MRP.GridInventory.inventories;
	contents.dropInventories   = MRP.GridInventory.dropInventories;
    return contents;
};

MRP.GridInventory.DataManager_extractSaveContents = DataManager.extractSaveContents;
DataManager.extractSaveContents = function(contents) {
    MRP.GridInventory.DataManager_extractSaveContents.call(this, contents);
    MRP.GridInventory.inventories       = contents.inventories;
	MRP.GridInventory.dropInventories   = contents.dropInventories;
};

Game_Map.prototype.makeDropEvent = function(items) {
	$gameTemp._triedDropEvents = $gameTemp._triedDropEvents != undefined ? $gameTemp._triedDropEvents : [];
	var foundEvent = false;

	// Check if there is a valid event already
	for(var i = 0; i < this.events().length; i++) {
		var event = this.events()[i];
		if(!$gameTemp._triedDropEvents.includes(i) && event._inventoryId != undefined && event._eventX == $gamePlayer.x && event._eventY == $gamePlayer.y) {
			$gameTemp._triedDropEvents.push(i);
			if(event._erased) {
				event._erased = false;
				event.refresh();
			}
			event.pushItems(items);
			foundEvent = true;
			break;
		}
	}

	// Make new event
	if(!foundEvent) {
		var event = new Game_DropEvent(this.events().length + 1, this._mapId, items)
		this._events.push(event);
		if(SceneManager._scene.constructor == Scene_Map) {
			SceneManager._scene._spriteset._tilemap.addChild(new Sprite_Character(event));
		}
	}
	// Remake the lower layer
	$gameTemp._triedDropEvents = [];
}

DataManager.makeDropInventory = function(id, cols, rows) {
	var inventory = new Game_Grid("Drops", cols, rows, false);
	MRP.GridInventory.dropInventories[id] = inventory;
	return inventory;
}

DataManager.removeDropInventory = function(id) {
	MRP.GridInventory.dropInventories[id] = undefined;
}

DataManager.getMaxItems = function(item){
	var value = 1;
	if(item) {
		if(item.meta.maxItems){
			value = item.meta.maxItems;
		} else {
			if(Imported.YEP_ItemCore){
				//value = Game_Party.prototype.getIndependentItemTypeMax.call(this, item)
			}
		}
	}
	return value > 0 ? value : 1;
}